#!/bin/bash

# Set this to y to turn on debugging in this script.
#DEBUG=y

###############################################
# _test_config_var
# pass in
# $1 - config variable to check
# $2 - Current parse state, 0 = OK, 1 = Failed
# return parse state on absence or presence of $1.

function _test_config_var () {
    local CONFIG_VAR="${1}"
    local CONFIG_ERR=${2}

    # debug test
    if [[ ! -z "${DEBUG}" ]] ; then
        echo "+=+ ${CONFIG_ERR} - Checking ${CONFIG_VAR}"
    fi

    # perform test
    if [[ -z $(eval "echo \"\$${CONFIG_VAR}\"") ]] ; then
        CONFIG_ERR=1
    fi
    return ${CONFIG_ERR}
}

# Directory of this script, hence the directory of the hsoa_create_version_include repository
OSDEV_GIT_DIR="$(cd $(dirname ${0}); pwd)"

# What language to generate for
OSDEV_VERSION_TYPE="C++"

# Where the component/product git repository is, default to where this script is.
OSDEV_PRODUCT_SRC_DIR="${OSDEV_PRODUCT_SRC_DIR:=$1}"

# configuration file directory (default to current environment)
OSDEV_CONFIG_DIR="${OSDEV_CONFIG_DIR:=${OSDEV_PRODUCT_SRC_DIR}/osdev_config}"

# path to customer configuration file
OSDEV_VERSION_CUSTOMER_CONFIG="${OSDEV_VERSION_CUSTOMER_CONFIG:=${OSDEV_GIT_DIR}/osdev_config/customer_config.txt}"

# Architecture version filename
OSDEV_VERSION_OSDEV_CONFIG="${OSDEV_VERSION_OSDEV_CONFIG:=${OSDEV_GIT_DIR}/osdev_config/osdev_config.txt}"

# product configuration filename
OSDEV_VERSION_PRODUCT_CONFIG="${OSDEV_VERSION_PRODUCT_CONFIG:=${OSDEV_CONFIG_DIR}/product_config.txt}"

# output header directory (default to environment)
OSDEV_VERSION_HEADER_DIR="${OSDEV_VERSION_HEADER_DIR:=$2}"

# filename to create
OSDEV_VERSION_FILE_PREFIX=${OSDEV_VERSION_HEADER_DIR}/osdev_version

if [[ ! -z "${DEBUG}" ]] ; then
    echo "OSDEV_GIT_DIR=${OSDEV_GIT_DIR}"
    echo "OSDEV_CONFIG_DIR=${OSDEV_CONFIG_DIR}"
    echo "OSDEV_PRODUCT_SRC_DIR=${OSDEV_PRODUCT_SRC_DIR}"
    echo "OSDEV_VERSION_CUSTOMER_CONFIG=${OSDEV_VERSION_CUSTOMER_CONFIG}"
    echo "OSDEV_VERSION_PRODUCT_CONFIG=${OSDEV_VERSION_PRODUCT_CONFIG}"
    echo "OSDEV_VERSION_OSDEV_CONFIG=${OSDEV_VERSION_OSDEV_CONFIG}"
    echo "OSDEV_VERSION_HEADER_DIR=${OSDEV_VERSION_HEADER_DIR}"
    echo "OSDEV_VERSION_FILE_PREFIX=${OSDEV_VERSION_FILE_PREFIX}"
fi

# find local C compiler to test header file
if [[ -z "${CC}" ]] ; then
    CC="$(which cc)"
fi

# no error found yet
CONFIG_ERR=0

# check osdev_config.txt for architecture configuration
if [[ ! -e ${OSDEV_VERSION_OSDEV_CONFIG} ]] ; then
    CONFIG_ERR=1
else
    if [[ ! -z "${DEBUG}" ]] ; then
        echo "+=+ Debug - parse for ${OSDEV_VERSION_OSDEV_CONFIG}"
    fi
    source ${OSDEV_VERSION_OSDEV_CONFIG}
    CONFIG_ERR= _test_config_var "OSDEV_IDENTIFIER" "${CONFIG_ERR}"
    CONFIG_ERR= _test_config_var "OSDEV_RELEASE_NAME" "${CONFIG_ERR}"
fi

if [[ 0 -ne ${CONFIG_ERR} ]] ; then
    echo "-- FATAL: osdev_config.txt ${OSDEV_VERSION_OSDEV_CONFIG} does not exist or is corrupt"
    exit 1
else
    echo "++ osdev_hsoa:${OSDEV_RELEASE_NAME}_${OSDEV_IDENTIFIER}"
fi

# check for a product configuration
if [[ ! -e ${OSDEV_VERSION_PRODUCT_CONFIG} ]] ; then
    CONFIG_ERR=1
else
    if [[ ! -z "${DEBUG}" ]] ; then
        echo "+=+ Debug - parse for ${OSDEV_VERSION_PRODUCT_CONFIG}"
    fi
    source ${OSDEV_VERSION_PRODUCT_CONFIG}
    # check required parameters are there
    CONFIG_ERR= _test_config_var "PRODUCT_RELEASE_NAME" "${CONFIG_ERR}"
    CONFIG_ERR= _test_config_var "PRODUCT_MANUFACTURER_CODE" "${CONFIG_ERR}"
fi

if [[ 0 -ne ${CONFIG_ERR} ]] ; then
    echo "-- FATAL: product_config.txt ${OSDEV_VERSION_PRODUCT_CONFIG} does not exist or is corrupt"
    exit 1
else
    echo "++ product:${PRODUCT_MANUFACTURER_CODE}-${PRODUCT_RELEASE_NAME}"
fi

CUSTOMER_ID=""
if [[ ! -e ${OSDEV_VERSION_CUSTOMER_CONFIG} ]] ; then
    CONFIG_ERR=1
else
    if [[ ! -z "${DEBUG}" ]] ; then
        echo "+=+ Debug - parse for ${OSDEV_VERSION_CUSTOMER_CONFIG}"
    fi
    source ${OSDEV_VERSION_CUSTOMER_CONFIG}
    # check required parameters are there
    CONFIG_ERR= _test_config_var  "CUSTOMER_NAME" "${CONFIG_ERR}"
    CONFIG_ERR= _test_config_var  "CUSTOMER_NUMBER" "${CONFIG_ERR}"
    CONFIG_ERR= _test_config_var  "CUSTOMER_VERSION" "${CONFIG_ERR}"

# do not put customer name in here, because when a plant is sold a name can be changed but
# customer number and version will not change, they are linked to the location
#    CUSTOMER_ID="${CUSTOMER_NUMBER}.${CUSTOMER_VERSION}_${CUSTOMER_NAME}"
    CUSTOMER_ID="${CUSTOMER_NUMBER}.${CUSTOMER_VERSION}"
fi

if [[ 0 -ne ${CONFIG_ERR} ]] ; then
    echo "-- FATAL: customer_config.txt ${OSDEV_VERSION_CUSTOMER_CONFIG} does not exist or is corrupt"
    exit 1
else
    echo "++ customer: ${CUSTOMER_ID}"
fi

VER_BUILD_ID_LONG_SHA="$(cd ${OSDEV_PRODUCT_SRC_DIR}; git rev-list HEAD | sed -e 's/\(.*\)/\1/;q;d')"
VER_BUILD_ID_SHORT_SHA="$(cd ${OSDEV_PRODUCT_SRC_DIR}; git rev-list HEAD | sed -e 's/\(........\).*/\1/;q;d')"
VER_BUILD_BRANCH="$(cd ${OSDEV_PRODUCT_SRC_DIR}; git name-rev --name-only HEAD)"
VER_BUILD_DATE="$(cd ${OSDEV_PRODUCT_SRC_DIR}; date +'%Y.%m.%d')"
VER_BUILD_DATETIME="$(cd ${OSDEV_PRODUCT_SRC_DIR}; date +'%Y.%m.%d.%T')"
VER_BUILD_TAG="$(cd ${OSDEV_PRODUCT_SRC_DIR}; git describe --abbrev=0 --tags)"
VER_BUILD_HOST="$(hostname -s)"

# in case version of McGraw+yocto is not defined set it to 1.0
if [ -z "$OSDEV_DISTRO_VERSION" ]; then
    OSDEV_DISTRO_VERSION="1.0"
fi

# split out tag information
IFS='.' read -ra VER_BUILD_VERSION <<< "$VER_BUILD_TAG"

VER_BUILD_MAJOR="${VER_BUILD_VERSION[0]}"
VER_BUILD_MINOR="${VER_BUILD_VERSION[1]}"
VER_BUILD_PATCH="${VER_BUILD_VERSION[2]}"
VER_BUILD_LETTER="${VER_BUILD_VERSION[3]}"
VER_BUILD_RELEASENR="1"

# set some defaults if git tag information is not available.
if [ -z "$VER_BUILD_MAJOR" ] ; then
    VER_BUILD_MAJOR="0"
fi
if [ -z "$VER_BUILD_MINOR" ] ; then
    VER_BUILD_MINOR="0"
fi
if [ -z "$VER_BUILD_PATCH" ] ; then
    VER_BUILD_PATCH="0"
fi
if [ -z "$VER_BUILD_LETTER" ] ; then
    # must be X,T,H,F,S or R,
    VER_BUILD_LETTER="X"
fi

# short version number
PRODUCT_RELEASE_SHORT_VERSION="${VER_BUILD_MAJOR}.${VER_BUILD_MINOR}"

# full version number
PRODUCT_RELEASE_VERSION="${VER_BUILD_MAJOR}.${VER_BUILD_MINOR}.${VER_BUILD_PATCH}.${VER_BUILD_LETTER}"

# full product version for product support, version number template:
PRODUCT_FULL_VERSION="${OSDEV_RELEASE_NAME}_${OSDEV_IDENTIFIER}_${PRODUCT_MANUFACTURER_CODE}-${PRODUCT_RELEASE_NAME}_${PRODUCT_RELEASE_VERSION}_${CUSTOMER_ID}_${VER_BUILD_ID_SHORT_SHA}_${VER_BUILD_DATETIME}"

# Replace spaces in customer name
CUSTOMER_NAME_STR=$(echo "${CUSTOMER_NAME}" | tr ' ' '_')

# Customer Configuration SHA
CUSTOMER_BUILD_FLAGS=$(echo $OSDEV_IMAGE_CONFIG_GITSHA)

VER_BUILD_EMBED_SHA=":git:${VER_BUILD_BRANCH}:${VER_BUILD_TAG}:${VER_BUILD_HOST}:${VER_BUILD_ID_LONG_SHA}:${PRODUCT_FULL_VERSION}:${PRODUCT_BUILD_FLAGS}:${CUSTOMER_BUILD_FLAGS}:tig:"

# echo extended version information to log/console
while read -r ECHO_LINE
do
    echo "${ECHO_LINE}"
done <<EOFOUT
-------------------------
${VER_BUILD_EMBED_SHA}
-------------------------
EOFOUT

# ensure include directory exists
mkdir -p ${OSDEV_VERSION_HEADER_DIR}

# generate a hsoa_version for cmake
while read -r ECHO_CMAKEFILE
do
    echo "${ECHO_CMAKEFILE}"
done <<EOF > ${OSDEV_VERSION_FILE_PREFIX}.cmake

# DO NOT EDIT THIS FILE!

# this file is auto-generated by create_version_include. The script that
# implements the "HSOA Component Lifecycle Management v1.x"
# component versioning standard.

# Set in parent scope because of add_subdirectory in hsoaversion.cmake.
set( OSDEV_VERSION_INCLUDE_DIR "${OSDEV_VERSION_HEADER_DIR}" PARENT_SCOPE )
set( OSDEV_VERSION_SRC_FILE "${OSDEV_VERSION_FILE_PREFIX}.cpp" PARENT_SCOPE )
set( CURRENT_PROJECT_MANUFACTURER_CODE "${PRODUCT_MANUFACTURER_CODE}" PARENT_SCOPE )
set( CURRENT_PROJECT_VERSION_MAJOR "${VER_BUILD_MAJOR}" PARENT_SCOPE )
set( CURRENT_PROJECT_VERSION_MINOR "${VER_BUILD_MINOR}" PARENT_SCOPE )
set( CURRENT_PROJECT_VERSION_PATCH "${VER_BUILD_PATCH}" PARENT_SCOPE )
set( CURRENT_PROJECT_VERSION_LETTER "${VER_BUILD_LETTER}" PARENT_SCOPE )
set( CURRENT_PROJECT_VERSION_RELEASENR "${VER_BUILD_RELEASENR}" PARENT_SCOPE )
set( CURRENT_PROJECT_VERSION "${VER_BUILD_MAJOR}.${VER_BUILD_MINOR}.${VER_BUILD_PATCH}" PARENT_SCOPE )

EOF

if [[ ! -z "${DEBUG}" ]] ; then
    cat ${OSDEV_VERSION_FILE_PREFIX}.cmake
fi

# generate a hsoa_version header file for c/c++
while read -r ECHO_HFILE
do
    echo "${ECHO_HFILE}"
done <<EOF > ${OSDEV_VERSION_FILE_PREFIX}.h

/*
DO NOT EDIT THIS FILE!

this file is auto-generated by create_version_include. The script that
implements the "HSOA Component Lifecycle Management v1.x"
component versioning standard.

*/

#ifndef OSDEV_VERSION_DEFINES_H_
#define OSDEV_VERSION_DEFINES_H_

namespace mlogic {
namespace hsoaversion {
namespace PRODUCT_${PRODUCT_RELEASE_NAME} {

#if !defined(OSDEV_GNUC_USED) && (defined(__GNUC__) && !defined(__clang__))
    #define OSDEV_GNUC_USED [[gnu::used]]
#else
    #define OSDEV_GNUC_USED
#endif

struct HsoaVersionInfo
{
/* mlogic specific information */
OSDEV_GNUC_USED static const char* VER_OSDEV_RELEASE_NAME_;
OSDEV_GNUC_USED static const char* VER_OSDEV_IDENTIFIER_;

/* customer specific information */
OSDEV_GNUC_USED static const char* VER_CUSTOMER_NAME_;
OSDEV_GNUC_USED static const char* VER_CUSTOMER_NUMBER_;
OSDEV_GNUC_USED static const char* VER_CUSTOMER_OSDEV_VERSION_;
OSDEV_GNUC_USED static const char* VER_CUSTOMER_ID_;

/* product specific information */
OSDEV_GNUC_USED static const char* VER_PRODUCT_RELEASE_NAME_;
OSDEV_GNUC_USED static const char* VER_PRODUCT_RELEASE_VSHORT_;
OSDEV_GNUC_USED static const char* VER_PRODUCT_RELEASE_VFULL_;
OSDEV_GNUC_USED static const char* VER_PRODUCT_OSDEV_VERSION_;
OSDEV_GNUC_USED static const char* VER_PRODUCT_MANUFACTURER_CODE_;
OSDEV_GNUC_USED static const char* VER_PRODUCT_MANUFACTURER_URL_;

/* version major, minor, sub and letter. */
OSDEV_GNUC_USED static const char* VER_BUILD_MAJOR_;
OSDEV_GNUC_USED static const char* VER_BUILD_MINOR_;
OSDEV_GNUC_USED static const char* VER_BUILD_PATCH_;
OSDEV_GNUC_USED static const char* VER_BUILD_RELEASENR_;
OSDEV_GNUC_USED static const char* VER_BUILD_LETTER_;

/* Git build specific information to embed into the program */
OSDEV_GNUC_USED static const char* VER_BUILD_ID_SHORT_SHA_;
OSDEV_GNUC_USED static const char* VER_BUILD_ID_LONG_SHA_;
OSDEV_GNUC_USED static const char* VER_BUILD_DATE_;
OSDEV_GNUC_USED static const char* VER_BUILD_DATETIME_;
OSDEV_GNUC_USED static const char* VER_BUILD_EMBED_SHA_;
OSDEV_GNUC_USED static const char* VER_BUILD_CUSTOMER_FLAGS_;
OSDEV_GNUC_USED static const char* VER_BUILD_PRODUCT_FLAGS_;
};

} // namespace PRODUCT_...
} // namespace hsoaversion
} // namespace mlogic

#endif /* OSDEV_VERSION_DEFINES_H_ */

EOF

if [[ ! -z "${DEBUG}" ]] ; then
    cat ${OSDEV_VERSION_FILE_PREFIX}.h
fi

# generate a hsoa_version source file for c/c++
while read -r ECHO_CPPFILE
do
    echo "${ECHO_CPPFILE}"
done <<EOF > ${OSDEV_VERSION_FILE_PREFIX}.cpp

/*
DO NOT EDIT THIS FILE!

this file is auto-generated by create_version_include. The script that
implements the "HSOA Component Lifecycle Management v1.x"
component versioning standard.

*/

#include "hsoa_version.h"

using namespace mlogic::hsoaversion::PRODUCT_${PRODUCT_RELEASE_NAME};

/* mlogic specific information */
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_OSDEV_RELEASE_NAME_= "${OSDEV_RELEASE_NAME}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_OSDEV_IDENTIFIER_    = "${OSDEV_IDENTIFIER}";

/* customer specific information */
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_CUSTOMER_NAME_        = "${CUSTOMER_NAME_STR}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_CUSTOMER_NUMBER_      = "${CUSTOMER_NUMBER}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_CUSTOMER_OSDEV_VERSION_= "${CUSTOMER_VERSION}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_CUSTOMER_ID_          = "${CUSTOMER_ID}";

/* product specific information */
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_PRODUCT_RELEASE_NAME_  = "${PRODUCT_RELEASE_NAME}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_PRODUCT_RELEASE_VSHORT_= "${PRODUCT_RELEASE_SHORT_VERSION}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_PRODUCT_RELEASE_VFULL_ = "${PRODUCT_RELEASE_VERSION}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_PRODUCT_OSDEV_VERSION_  = "${PRODUCT_FULL_VERSION}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_PRODUCT_MANUFACTURER_CODE_ = "${PRODUCT_MANUFACTURER_CODE}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_PRODUCT_MANUFACTURER_URL_ = "${PRODUCT_MANUFACTURER_URL}";

/* version major, minor, sub and letter. */
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_MAJOR_  = "${VER_BUILD_MAJOR}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_MINOR_  = "${VER_BUILD_MINOR}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_PATCH_  = "${VER_BUILD_PATCH}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_RELEASENR_ = "${VER_BUILD_RELEASENR}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_LETTER_ = "${VER_BUILD_LETTER}";

/* Git build specific information to embed into the program */
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_ID_SHORT_SHA_  = "${VER_BUILD_ID_SHORT_SHA}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_ID_LONG_SHA_   = "${VER_BUILD_ID_LONG_SHA}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_DATE_          = "${VER_BUILD_DATE}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_DATETIME_      = "${VER_BUILD_DATETIME}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_EMBED_SHA_     = "${VER_BUILD_EMBED_SHA}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_CUSTOMER_FLAGS_= "${CUSTOMER_BUILD_FLAGS}";
OSDEV_GNUC_USED const char* HsoaVersionInfo::VER_BUILD_PRODUCT_FLAGS_ = "${PRODUCT_BUILD_FLAGS}";

EOF

if [[ ! -z "${DEBUG}" ]] ; then
    cat ${OSDEV_VERSION_FILE_PREFIX}.h
fi

# generate a hsoa_version for python
while read -r ECHO_HFILE
do
    echo "${ECHO_HFILE}"
done <<EOF > ${OSDEV_VERSION_FILE_PREFIX}.py

#
# DO NOT EDIT THIS FILE!
#
# this file is auto-generated by create_version_include. The script that
# implements the "HSOA Component Lifecycle Management v1.x"
# component versioning standard.
#
#

# mlogic specific information
VER_OSDEV_RELEASE_NAME_ = "${OSDEV_RELEASE_NAME}"
VER_OSDEV_IDENTIFIER_ = "${OSDEV_IDENTIFIER}"

# customer specific information
VER_CUSTOMER_NAME_ = "${CUSTOMER_NAME_STR}"
VER_CUSTOMER_NUMBER_ = "${CUSTOMER_NUMBER}"
VER_CUSTOMER_OSDEV_VERSION_ = "${CUSTOMER_VERSION}"
VER_CUSTOMER_ID_ = "${CUSTOMER_ID}"

# product specific information
VER_PRODUCT_RELEASE_NAME_ = "${PRODUCT_RELEASE_NAME}"
VER_PRODUCT_RELEASE_VSHORT_ = "${PRODUCT_RELEASE_SHORT_VERSION}"
VER_PRODUCT_RELEASE_VFULL_ = "${PRODUCT_RELEASE_VERSION}"
VER_PRODUCT_OSDEV_VERSION_ = "${PRODUCT_FULL_VERSION}"
VER_PRODUCT_MANUFACTURER_CODE_ = "${PRODUCT_MANUFACTURER_CODE}"

# version major, minor, sub and letter.
VER_BUILD_MAJOR_ = "${VER_BUILD_MAJOR}"
VER_BUILD_MINOR_ = "${VER_BUILD_MINOR}"
VER_BUILD_PATCH_ = "${VER_BUILD_PATCH}"
VER_BUILD_LETTER_ = "${VER_BUILD_LETTER}"

# Git build specific information to embed into the program
VER_BUILD_ID_SHORT_SHA_ = "${VER_BUILD_ID_SHORT_SHA}"
VER_BUILD_ID_LONG_SHA_ = "${VER_BUILD_ID_LONG_SHA}"
VER_BUILD_DATE_ = "${VER_BUILD_DATE}"
VER_BUILD_DATETIME_ = "${VER_BUILD_DATETIME}"
VER_BUILD_EMBED_SHA_ = "${VER_BUILD_EMBED_SHA}"
VER_BUILD_CUSTOMER_FLAGS_ = "${CUSTOMER_BUILD_FLAGS}"
VER_BUILD_PRODUCT_FLAGS_ = "${PRODUCT_BUILD_FLAGS}"

EOF

if [[ ! -z "${DEBUG}" ]] ; then
    cat ${OSDEV_VERSION_FILE_PREFIX}.py
fi

# test python output
if [[ ! -z "$(which pylint)" ]] ; then
    echo "++ testing ${OSDEV_VERSION_FILE_PREFIX}.py"
    pylint -E ${OSDEV_VERSION_FILE_PREFIX}.py
else
    echo "++ pylint not available to test ${OSDEV_VERSION_FILE_PREFIX}.py"
fi

# generate a hsoa_version for JavaScript
while read -r ECHO_HFILE
do
    echo "${ECHO_HFILE}"
done <<EOF > ${OSDEV_VERSION_FILE_PREFIX}.js

/*
DO NOT EDIT THIS FILE!

this file is auto-generated by create_version_include. The script that
implements the "HSOA Component Lifecycle Management v1.x"
component versioning standard.

*/

/* mlogic specific information */
const VER_OSDEV_RELEASE_NAME_ = "${OSDEV_RELEASE_NAME}"
const VER_OSDEV_IDENTIFIER_     = "${OSDEV_IDENTIFIER}"

/* customer specific information */
const VER_CUSTOMER_NAME_         = "${CUSTOMER_NAME_STR}"
const VER_CUSTOMER_NUMBER_       = "${CUSTOMER_NUMBER}"
const VER_CUSTOMER_OSDEV_VERSION_ = "${CUSTOMER_VERSION}"
const VER_CUSTOMER_ID_           = "${CUSTOMER_ID}"

/* product specific information */
const VER_PRODUCT_RELEASE_NAME_   = "${PRODUCT_RELEASE_NAME}"
const VER_PRODUCT_RELEASE_VSHORT_ = "${PRODUCT_RELEASE_SHORT_VERSION}"
const VER_PRODUCT_RELEASE_VFULL_  = "${PRODUCT_RELEASE_VERSION}"
const VER_PRODUCT_OSDEV_VERSION_   = "${PRODUCT_FULL_VERSION}"
const VER_PRODUCT_MANUFACTURER_CODE_  = "${PRODUCT_MANUFACTURER_CODE}"

/* version major, minor, sub and letter. */
const VER_BUILD_MAJOR_  = "${VER_BUILD_MAJOR}"
const VER_BUILD_MINOR_  = "${VER_BUILD_MINOR}"
const VER_BUILD_PATCH_    = "${VER_BUILD_PATCH}"
const VER_BUILD_LETTER_ = "${VER_BUILD_LETTER}"

/* Git build specific information to embed into the program */
const VER_BUILD_ID_SHORT_SHA_   = "${VER_BUILD_ID_SHORT_SHA}"
const VER_BUILD_ID_LONG_SHA_    = "${VER_BUILD_ID_LONG_SHA}"
const VER_BUILD_DATE_           = "${VER_BUILD_DATE}"
const VER_BUILD_DATETIME_       = "${VER_BUILD_DATETIME}"
const VER_BUILD_EMBED_SHA_      = "${VER_BUILD_EMBED_SHA}"
const VER_BUILD_CUSTOMER_FLAGS_ = "${CUSTOMER_BUILD_FLAGS}"
const VER_BUILD_PRODUCT_FLAGS_  = "${PRODUCT_BUILD_FLAGS}"

EOF

if [[ ! -z "${DEBUG}" ]] ; then
    cat ${OSDEV_VERSION_FILE_PREFIX}.js
fi

# test JavaScript output with node
if [[ ! -z "$(which node)" ]] ; then
    echo "++ testing ${OSDEV_VERSION_FILE_PREFIX}.js"
    node ${OSDEV_VERSION_FILE_PREFIX}.js
else
    echo "++ node not available to test ${OSDEV_VERSION_FILE_PREFIX}.js"
fi
