/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

#ifndef OSDEV_COMPONENTS_TRANSQUEUE_H
#define OSDEV_COMPONENTS_TRANSQUEUE_H

#include "ormreldata.h"

#include <QObject>
#include <QQueue>
#include <QTimer>
#include <QMutex>
#include <QMutexLocker>

/*
 *  _________________________________________
 * / If voting could change the system, it   \
 * | would be illegal. If not voting could   |
 * \ change the system, it would be illegal. /
 *  -----------------------------------------
 *    \
 *     \
 *         .--.
 *        |o_o |
 *        |:_/ |
 *       //   \ \
 *      (|     | )
 *     /'\_   _/`\
 *     \___)=(___/
 *
 */

namespace osdev {
namespace components {
/*!
 * \class   TransQueue
 * \brief   The TransQueue class implements a transaction queue which will
 *          catch rejected transactions from the ORM layer and offers them
 *          back through the ETL-layer. An ORMRelPackage is "tagged" with
 *          a transacion-id, which will keep track on how many times an
 *          ORMRelPackage enters the queue. If the "TTL_iD (Time_To_live)"
 *          reaches zero, the package is dropped and the ID is removed from
 *          the administration.
 */
class TransQueue : public QObject
{
    Q_OBJECT

public:
    /*!
     * \brief Represents a queue of transactions.
     * \param parent The parent QObject instance.
     */
    explicit TransQueue( QObject *parent = nullptr );

    // Deleted copy- and move constructors
    TransQueue( const TransQueue& ) = delete;
    TransQueue( const TransQueue&& ) = delete;
    TransQueue& operator=( const TransQueue& ) = delete;
    TransQueue& operator=( const TransQueue&& ) = delete;

    /*!
     * \brief Sets the TimeOut.
     * \param milliseconds The timeout to set in milliseconds.
     */
    void    setTimeOut( int milliseconds = 1 );

    /*!
     * \return Whether this instance is currently processing transactions.
     */
    bool    processing() const;

    /*!
     * \brief Starts processing of transactions.
     * \param force Forces start of processing.
     */
    void    startProcessing( bool force = false );

    /*!
     * \brief Stops processing of transactions.
     * \param force Forces stop of processing.
     */
    void    stopProcessing( bool force = false );

    /*!
     * \return The number of currently outstanding transactions.
     */
    int     transactions() const { return m_queue.size(); }

private:
    /*!
     * \brief   This method will write the Contents of the data structure
     *          to a transaction file for archiving and traceability purposes.
     *          Not really decided on the format yet, but this will evolve over time.
     * \param   data    - The datastructure we want to write to a transaction Logfile.
     *                    Send as pointer. After writing to disk, the object have to be deleted.
     */
    bool dumpToDisk( const QSharedPointer<ORMRelData>& data ) const;

    int     m_TTL;                              ///< Time to Live start value.
    QTimer  m_queueTimer;                       ///< Timer controlling the processing of the queue.
    QMutex  m_queueMutex;                       ///< Mutex to prevent race conditions. Although QQueue is thread safe, we have to make sure *we* are.
    QQueue<QSharedPointer<ORMRelData>> m_queue; ///< The actual FIFO taking an ORMRelData pointer as input.

signals:
    void    signalProcessData( QSharedPointer<ORMRelData> pData );

public slots:
    /*!
     * \brief Sets (adds) a transaction to the queue.
     * \param pData The transaction data to set.
     */
    void setTransaction( const QSharedPointer<ORMRelData>& pData );


private slots:
    /*!
     * \brief slotProcessQueue
     */
    void    slotProcessQueue();
};

}   /* End namespace components */
}   /* End namespace osdev */

#endif  /* OSDEV_COMPONENTS_TRANSQUEUE_H */
