#include "socket-cpp/inet_address.h"
#include "socket-cpp/exception.h"

using namespace std;
using namespace osdev::components::socket-cpp;

bool inet_address::is_set() const
{
	static const auto EMPTY_ADDR = sockaddr_in{};
	return std::memcmp(&addr_, &EMPTY_ADDR, SZ) != 0;
}

in_addr_t inet_address::resolve_name(const std::string& saddr)
{
	#if !defined(_WIN32)
		in_addr ia;
		if (::inet_pton(ADDRESS_FAMILY, saddr.c_str(), &ia) == 1)
			return ia.s_addr;
	#endif

    addrinfo *res, hints = addrinfo{};
    hints.ai_family = ADDRESS_FAMILY;
    hints.ai_socktype = SOCK_STREAM;

    int gai_err = ::getaddrinfo(saddr.c_str(), NULL, &hints, &res);

    #if !defined(_WIN32)
        if (gai_err == EAI_SYSTEM)
            throw sys_error();
    #endif

    if (gai_err != 0)
        throw getaddrinfo_error(gai_err, saddr);

    auto ipv4 = reinterpret_cast<sockaddr_in*>(res->ai_addr);
    auto addr = ipv4->sin_addr.s_addr;
    freeaddrinfo(res);
    return addr;
}

void inet_address::create(uint32_t addr, in_port_t port)
{
	addr_ = sockaddr_in{};
	addr_.sin_family = AF_INET;
	addr_.sin_addr.s_addr = htonl(addr);
	addr_.sin_port = htons(port);
}

void inet_address::create(const std::string& saddr, in_port_t port)
{
	addr_ = sockaddr_in{};
	addr_.sin_family = AF_INET;
	addr_.sin_addr.s_addr = resolve_name(saddr.c_str());
	addr_.sin_port = htons(port);
}

string inet_address::to_string() const
{
	char buf[INET_ADDRSTRLEN];
	auto str = inet_ntop(AF_INET, (void*) &(addr_.sin_addr), buf, INET_ADDRSTRLEN);
	return std::string(str ? str : "<unknown>")
		+ ":" + std::to_string(unsigned(port()));
}

ostream& operator<<(ostream& os, const inet_address& addr)
{
    char buf[INET_ADDRSTRLEN];
	auto str = inet_ntop(AF_INET, (void*) &(addr.sockaddr_in_ptr()->sin_addr),
						 buf, INET_ADDRSTRLEN);
	os << (str ? str : "<unknown>") << ":" << unsigned(addr.port());
	return os;
}

