#include "socket-cpp/inet6_address.h"
#include "socket-cpp/exception.h"

using namespace std;
using namespace osdev::components::socket-cpp;

bool inet6_address::is_set() const
{
	static const auto EMPTY_ADDR = sockaddr_in6{};
	return std::memcmp(&addr_, &EMPTY_ADDR, SZ) != 0;
}

in6_addr inet6_address::resolve_name(const string& saddr)
{
	#if !defined(_WIN32)
		in6_addr ia;
		if (::inet_pton(ADDRESS_FAMILY, saddr.c_str(), &ia) == 1)
			return ia;
	#endif

    addrinfo *res, hints = addrinfo{};
    hints.ai_family = ADDRESS_FAMILY;
    hints.ai_socktype = SOCK_STREAM;

    int gai_err = ::getaddrinfo(saddr.c_str(), NULL, &hints, &res);

    #if !defined(_WIN32)
        if (gai_err == EAI_SYSTEM)
            throw sys_error();
    #endif

    if (gai_err != 0)
        throw getaddrinfo_error(gai_err, saddr);


    auto ipv6 = reinterpret_cast<sockaddr_in6*>(res->ai_addr);
    auto addr = ipv6->sin6_addr;
    freeaddrinfo(res);
    return addr;
}

void inet6_address::create(const in6_addr& addr, in_port_t port)
{
	addr_ = sockaddr_in6{};
    addr_.sin6_family = AF_INET6;
    addr_.sin6_flowinfo = 0;
    addr_.sin6_addr = addr;
    addr_.sin6_port = htons(port);
}

void inet6_address::create(const string& saddr, in_port_t port)
{
	addr_ = sockaddr_in6{};
	addr_.sin6_family = AF_INET6;
    addr_.sin6_flowinfo = 0;
	addr_.sin6_addr = resolve_name(saddr.c_str());
	addr_.sin6_port = htons(port);
}

string inet6_address::to_string() const
{
    char buf[INET6_ADDRSTRLEN];
    auto str = inet_ntop(AF_INET6, (void*) &(addr_.sin6_addr),
						 buf, INET6_ADDRSTRLEN);
    return std::string("[") + std::string(str ? str : "<unknown>")
        + "]:" + std::to_string(unsigned(port()));
}

ostream& operator<<(ostream& os, const inet6_address& addr)
{
	char buf[INET6_ADDRSTRLEN];
	auto str = inet_ntop(AF_INET6, (void*) &(addr.sockaddr_in6_ptr()->sin6_addr),
						 buf, INET6_ADDRSTRLEN);
	os << "[" << (str ? str : "<unknown>") << "]:" << unsigned(addr.port());
	return os;
}

