#pragma once

#include <cerrno>
#include <stdexcept>
#include <string>

namespace osdev {
namespace components {
namespace socket-cpp {

/**
 * System error.
 * These are errors that are resulted from system socket calls. The error
 * codes are platform 'errno' values (or similar), and the messages are
 * typically derived from the system.
 */
class sys_error : public std::runtime_error
{
	/** The system error number (errno) */
	int errno_;

public:
	/**
	 * Creates an error using the current system 'errno' value.
	 */
	sys_error() : sys_error(errno) {}
	/**
	 * Constructs an error with the specified system errno.
	 * @param err The error number. This is the system errno value.
	 */
	explicit sys_error(int err);
	/**
	 * Get the error number.
	 * @return The system error number.
	 */
	int error() const { return errno_; }
    /**
     * Gets a string describing the specified error.
     * This is typically the returned message from the system strerror().
     * @param err The system error number.
     * @return A string describing the specified error.
     */
    static std::string error_str(int err);
};

/**
 * Errors from getaddrinfo.
 * These are errors relating to DNS lookup, returned from the getaddrinfo system call.
 * Their codes are declared in <netdb.h>.
 */
class getaddrinfo_error : public std::runtime_error
{
	/** The error number, as returned by getaddrinfo. */
	int error_;
    /** The hostname being resolved. */
    std::string hostname_;

public:
	/**
	 * Constructs an error with the specified getaddrinfo error code.
	 * @param err The error number, as returned by getaddrinfo.
     * @param hostname The DNS name being resolved that triggered the error.
	 */
	getaddrinfo_error(int err, const std::string &hostname);
	/**
	 * Get the error number.
	 * @return The error number returned by getaddrinfo.
	 */
	int error() const { return error_; }
    /**
     * Get the hostname that triggered the error.
     * @return The hostname that getaddrinfo failed to resolve.
     */
    const std::string &hostname() const { return hostname_; }
};

}   // End namespace socket-cpp
}   // End namespace components
}   // End namespace osdev

