#pragma once

#include "socket-cpp/datagram_socket.h"
#include "socket-cpp/can_address.h"
#include "socket-cpp/can_frame.h"

namespace osdev {
namespace components {
namespace socket-cpp {

/**
 * Socket type for Linux SocketCAN.
 *
 * Note that technically these are RAW sockets, not DGRAM. We can/should
 * organize the underlying hierarch to properly indicate this, but for
 * practical usge, it doesn't matter too MUCH.
 * The BCM CAN sockets are DGRAM sockets, but those aren't implemented yet.
 * It wouldn't take too much to add them, though.
 */
class can_socket : public datagram_socket
{
	/** The base class */
	using base = datagram_socket;

	// Non-copyable
	can_socket(const can_socket&) =delete;
	can_socket& operator=(const can_socket&) =delete;

	/**
	 * We can't connect to a raw CAN socket;
	 * we can only bind the address/iface.
	 */
	bool connect(const sock_address&) =delete;

protected:
	static socket_t create_handle(int type, int protocol) {
		return socket_t(::socket(PROTOCOL_FAMILY, type, protocol));
	}

public:
	/**
	 *  The SocketCAN protocol family.
	 *  Note that AF_CAN == PF_CAN, which is used in many of the CAN
	 *  examples.
	 */
	static const int PROTOCOL_FAMILY = AF_CAN;

	/** The socket 'type' for communications semantics. */
	static constexpr int COMM_TYPE = SOCK_RAW;

	/**
	 * Creates an uninitialized CAN socket.
	 */
	can_socket() {}
	/**
     * Creates a CAN socket from an existing OS socket handle and
     * claims ownership of the handle.
	 * @param handle A socket handle from the operating system.
	 */
	explicit can_socket(socket_t handle) : base(handle) {}
	/**
	 * Creates a CAN socket and binds it to the address.
	 * @param addr The address to bind.
	 */
	explicit can_socket(const can_address& addr);
	/**
	 * Move constructor.
	 * @param other The other socket to move to this one
	 */
	can_socket(can_socket&& other) : base(std::move(other)) {}
	/**
	 * Move assignment.
	 * @param rhs The other socket to move into this one.
	 * @return A reference to this object.
	 */
	can_socket& operator=(can_socket&& rhs) {
		base::operator=(std::move(rhs));
		return *this;
	}

	/**
	 * Gets the system time of the last frame read from the socket.
	 * @return The system time of the last frame read from the socket with
	 *  	   microsecond precision.
	 */
	std::chrono::system_clock::time_point last_frame_time();
	/**
	 * Gets a floating point timestamp of the last frame read from the
	 * socket.
	 * This is the number of seconds since the Unix epoch (time_t), with
	 * floating-point, microsecond precision.
	 * @return A floating-point timestamp with microsecond precision.
	 */
	double last_frame_timestamp();

	// ----- I/O -----

	/**
	 * Sends a frame to the CAN interfacce at the specified address.
	 * @param frame The CAN frame to send.
	 * @param flags The flags. See send(2).
	 * @param addr The remote destination of the data.
	 * @return the number of bytes sent on success or, @em -1 on failure.
	 */
	ssize_t send_to(const can_frame& frame, int flags, const can_address& addr) {
		return check_ret(
			::sendto(handle(), &frame, sizeof(can_frame), flags,
					 addr.sockaddr_ptr(), addr.size())
	    );
	}

	/**
	 * Sends a frame to the CAN interface at the specified address.
	 * @param frame The CAN frame to send.
	 * @param addr The remote destination of the data.
	 * @return the number of bytes sent on success or, @em -1 on failure.
	 */
	ssize_t send_to(const can_frame& frame, const can_address& addr) {
		return check_ret(
			::sendto(handle(), &frame, sizeof(can_frame), 0,
					 addr.sockaddr_ptr(), addr.size())
	    );
	}
	/**
	 * Sends a frame to the CAN bus.
	 * The socket should be bound before calling this.
	 * @param frame The CAN frame to send.
	 * @param flags The option bit flags. See send(2).
	 * @return @em zero on success, @em -1 on failure.
	 */
	ssize_t send(const can_frame& frame, int flags=0) {
		return check_ret(::send(handle(), &frame, sizeof(can_frame), flags));
	}
	/**
	 * Receives a message from the CAN interface at the specified address.
	 * @param frame CAN frame to get the incoming data.
	 * @param flags The option bit flags. See send(2).
	 * @param srcAddr Receives the address of the peer that sent the
	 *  			   message
	 * @return The number of bytes read or @em -1 on error.
	 */
	ssize_t recv_from(can_frame* frame, int flags, can_address* srcAddr=nullptr);
	/**
	 * Receives a message on the socket.
	 * @param frame CAN frame to get the incoming data.
	 * @param flags The option bit flags. See send(2).
	 * @return The number of bytes read or @em -1 on error.
	 */
	ssize_t recv(can_frame* frame, int flags=0) {
		return check_ret(::recv(handle(), frame, sizeof(can_frame), flags));
	}
};

}   // End namespace socket-cpp
}   // End namespace components
}   // End namespace osdev
