#include <iostream>
#include <string>
#include <chrono>
#include <random>
#include "socket-cpp/tcp_connector.h"
#include "socket-cpp/version.h"

using namespace std;
using namespace std::chrono;

const size_t DFLT_N = 100000;
const size_t DFLT_SZ = 512;

using fpsec = std::chrono::duration<double, std::chrono::seconds::period>;

// --------------------------------------------------------------------------

int main(int argc, char* argv[])
{
	cout << "Unix-domain echo timing test client for 'sockpp' "
		<< osdev::components::socket-cpp::SOCKPP_VERSION << '\n' << endl;

	string host = (argc > 1) ? argv[1] : "localhost";
	in_port_t port = (argc > 2) ? atoi(argv[2]) : 12345;

	size_t n  = (argc > 3) ? size_t(atoll(argv[3])) : DFLT_N;
	size_t sz = (argc > 4) ? size_t(atoll(argv[4])) : DFLT_SZ;

	osdev::components::socket-cpp::socket_initializer sockInit;

	auto t_start = high_resolution_clock::now();


	osdev::components::socket-cpp::tcp_connector conn({host, port});
	if (!conn) {
		cerr << "Error connecting to server at "
			<< osdev::components::socket-cpp::inet_address(host, port)
			<< "\n\t" << conn.last_error_str() << endl;
		return 1;
	}

	cout << "Created a connection from " << conn.address() << endl;
	cout << "Created a connection to " << conn.peer_address() << endl;

    // Set a timeout for the responses
    if (!conn.read_timeout(seconds(2))) {
        cerr << "Error setting timeout on TCP stream: "
                << conn.last_error_str() << endl;
    }
	string s, sret;

    random_device rd;
    mt19937 reng(rd());
    uniform_int_distribution<int> dist(0, 25);

	for (size_t i=0; i<sz; ++i)
		s.push_back('a' + static_cast<char>(dist(reng)));

	auto t_start_tx = high_resolution_clock::now();

	for (size_t i=0; i<n; ++i) {
		if (conn.write(s) != (ssize_t) s.length()) {
			cerr << "Error writing to the UNIX stream" << endl;
			break;
		}

		sret.resize(s.length());
		int n = conn.read_n(&sret[0], s.length());

		if (n != (int) s.length()) {
			cerr << "Error reading from UNIX stream" << endl;
			break;
		}
	}

	auto t_end = high_resolution_clock::now();
	cout << "Total time: " << fpsec(t_end - t_start).count() << "s" << endl;

	auto t_tx = fpsec(t_end - t_start_tx).count();
	auto rate = size_t(n / t_tx);
	cout << "Transfer time: " << t_tx << "s\n    "
		<< rate << " msg/s" << endl;

	return (!conn) ? 1 : 0;
}
