#include <iostream>
#include <string>
#include <thread>
#include "socket-cpp/tcp_connector.h"
#include "socket-cpp/version.h"

using namespace std;

// --------------------------------------------------------------------------
// The read thread will run independently, retrieving packets from the
// server and writing them to the console. When the main (write) thread
// shuts down the socket, we exit.

void read_thr(osdev::components::socket-cpp::tcp_socket rdSock)
{
	char buf[512];
	ssize_t n;

	while ((n = rdSock.read(buf, sizeof(buf))) > 0) {
		cout.write(buf, n);
		cout << endl;
	}

	if (n < 0) {
		cout << "Read error [" << rdSock.last_error() << "]: " 
			<< rdSock.last_error_str() << endl;
	}
	rdSock.shutdown();
}

// --------------------------------------------------------------------------

int main(int argc, char* argv[])
{
	cout << "Sample multi-threaded TCP echo client for 'sockpp' "
		<< osdev::components::socket-cpp::SOCKPP_VERSION << '\n' << endl;

	string host = (argc > 1) ? argv[1] : "localhost";
	in_port_t port = (argc > 2) ? atoi(argv[2]) : 12345;

	osdev::components::socket-cpp::socket_initializer sockInit;

	// Implicitly creates an inet_address from {host,port}
	// and then tries the connection.

	osdev::components::socket-cpp::tcp_connector conn({host, port});
	if (!conn) {
		cerr << "Error connecting to server at "
			<< osdev::components::socket-cpp::inet_address(host, port)
			<< "\n\t" << conn.last_error_str() << endl;
		return 1;
	}

	cout << "Created a connection from " << conn.address() << endl;

	// We create a read thread and send it a clone (dup) of the
	// connector socket. 

	std::thread rdThr(read_thr, std::move(conn.clone()));

	// The write loop get user input and writes it to the socket.

	string s, sret;
	while (getline(cin, s) && !s.empty()) {
		if (conn.write(s) != (int) s.length()) {
			if (conn.last_error() == EPIPE) {
				cerr << "It appears that the socket was closed." << endl;
			}
			else {
				cerr << "Error writing to the TCP stream ["
					<< conn.last_error() << "]: "
					<< conn.last_error_str() << endl;
			}
			break;
		}
	}
	int ret = !conn ? 1 : 0;

	// Shutting down the socket will cause the read thread to exit.
	// We wait for it to exit before we leave the app.

	conn.shutdown(SHUT_WR);
	rdThr.join();

	return ret;
}
