/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifdef QT_WEBSOCKETS_LIB

#include "qmqtt_websocket_p.h"

#include <QNetworkRequest>
#include <QUrl>
#include <QSslError>

#ifndef QT_NO_SSL
QMQTT::WebSocket::WebSocket(const QString& origin,
                            QWebSocketProtocol::Version version,
                            const QSslConfiguration* sslConfig,
                            QObject* parent)
    : SocketInterface(parent)
    , _socket(new QWebSocket(origin, version, this))
    , _ioDevice(new WebSocketIODevice(_socket, this))
{
    initialize();
    if (sslConfig != NULL)
        _socket->setSslConfiguration(*sslConfig);
    connect(_socket, &QWebSocket::sslErrors, this, &WebSocket::sslErrors);
}
#endif // QT_NO_SSL

QMQTT::WebSocket::WebSocket(const QString& origin,
                            QWebSocketProtocol::Version version,
                            QObject* parent)
    : SocketInterface(parent)
    , _socket(new QWebSocket(origin, version, this))
    , _ioDevice(new WebSocketIODevice(_socket, this))
{
    initialize();
}

void QMQTT::WebSocket::initialize()
{
    connect(_socket, &QWebSocket::connected, this, &WebSocket::connected);
    connect(_socket, &QWebSocket::disconnected, this, &WebSocket::disconnected);
    connect(_socket,
            static_cast<void (QWebSocket::*)(QAbstractSocket::SocketError)>(&QWebSocket::error),
            this,
            static_cast<void (SocketInterface::*)(QAbstractSocket::SocketError)>(&SocketInterface::error));
}

QMQTT::WebSocket::~WebSocket()
{
}

void QMQTT::WebSocket::connectToHost(const QHostAddress& address, quint16 port)
{
    Q_UNUSED(address)
    Q_UNUSED(port)
    qFatal("No supported");
}

void QMQTT::WebSocket::connectToHost(const QString& hostName, quint16 port)
{
    Q_UNUSED(port)
    QUrl url(hostName);
    QNetworkRequest request(url);
    request.setRawHeader("Sec-WebSocket-Protocol", "mqtt");
    _ioDevice->connectToHost(request);
}

void QMQTT::WebSocket::disconnectFromHost()
{
    _socket->close();
}

QAbstractSocket::SocketState QMQTT::WebSocket::state() const
{
    return _socket->state();
}

QAbstractSocket::SocketError QMQTT::WebSocket::error() const
{
    return _socket->error();
}

#ifndef QT_NO_SSL
void QMQTT::WebSocket::ignoreSslErrors(const QList<QSslError>& errors)
{
    _socket->ignoreSslErrors(errors);
}

void QMQTT::WebSocket::ignoreSslErrors()
{
    _socket->ignoreSslErrors();
}

QSslConfiguration QMQTT::WebSocket::sslConfiguration() const
{
    return _socket->sslConfiguration();
}

void QMQTT::WebSocket::setSslConfiguration(const QSslConfiguration& config)
{
    _socket->setSslConfiguration(config);
}

#endif // QT_NO_SSL

#endif // QT_WEBSOCKETS_LIB
