/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include "qmqtt_routesubscription.h"
#include "qmqtt_router.h"
#include "qmqtt_client.h"
#include "qmqtt_routedmessage.h"

#include <QLoggingCategory>
#include <QLatin1String>
#include <QLatin1Char>
#include <QRegularExpressionMatch>
#include <QStringList>

namespace QMQTT {

Q_LOGGING_CATEGORY(routerSubscription, "qmqtt.routersubscription")

RouteSubscription::RouteSubscription(Router *parent)
    : QObject(parent),
      _client(parent->client())
{
    Q_ASSERT(!_client.isNull());
}

RouteSubscription::~RouteSubscription()
{
    if (Q_LIKELY(!_client.isNull() && _client->isConnectedToHost()))
        _client->unsubscribe(_topic);
}

QString RouteSubscription::route() const
{
    return _topic;
}

void RouteSubscription::setRoute(const QString &route)
{
    QRegularExpression parameterNamesRegExp(QStringLiteral("\\:([a-zA-Z0-9]+)")); // note how names must not contain dashes or underscores

    // Remove paramter names to get the actual topic "route"
    QString topic = route;
    topic.remove(parameterNamesRegExp);

    // Remove the MQTT wildcards to get a regular expression, which matches the parameters
    QString parameterRegExp = route;
    parameterRegExp
            .remove(QLatin1Char('+'))
            .replace(parameterNamesRegExp, QStringLiteral("([a-zA-Z0-9_-]+)")) // note how parameter values may contain dashes or underscores
            .remove(QLatin1Char('#'))
            .replace(QLatin1String("$"), QLatin1String("\\$"));

    // Extract the parameter names
    QRegularExpressionMatchIterator it = parameterNamesRegExp.globalMatch(route);
    QStringList names;
    while(it.hasNext()) {
        QRegularExpressionMatch match = it.next();
        QString parameterName = match.captured(1);
        names << parameterName;
    }

    _topic = topic;
    _parameterNames = names;
    _regularExpression = QRegularExpression(parameterRegExp);
}

void RouteSubscription::routeMessage(const Message &message)
{
    QString topic = message.topic();
    QRegularExpressionMatch match = _regularExpression.match(topic);
    if(!match.hasMatch()) {
        return;
    }

    RoutedMessage routedMessage(message);

    for(int i = 0, c = _parameterNames.size(); i < c; ++i) {
        QString name = _parameterNames.at(i);
        QString value = match.captured(i + 1);

        routedMessage._parameters.insert(name, value);
    }

    emit received(routedMessage);
}

} // namespace QMQTT
