/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef QMQTT_NETWORK_P_H
#define QMQTT_NETWORK_P_H

#include "qmqtt_networkinterface.h"

#include <QObject>
#include <QHostAddress>
#include <QString>
#include <QByteArray>

#ifdef QT_WEBSOCKETS_LIB
#include <QWebSocket>
#endif // QT_WEBSOCKETS_LIB

#ifndef QT_NO_SSL
#include <QSslConfiguration>
QT_FORWARD_DECLARE_CLASS(QSslError)
#endif // QT_NO_SSL

namespace QMQTT {

class SocketInterface;
class TimerInterface;
class Frame;

class Network : public NetworkInterface
{
    Q_OBJECT

public:
    Network(QObject* parent = NULL);
#ifndef QT_NO_SSL
    Network(const QSslConfiguration& config, QObject* parent = NULL);
#endif // QT_NO_SSL
#ifdef QT_WEBSOCKETS_LIB
#ifndef QT_NO_SSL
    Network(const QString& origin,
            QWebSocketProtocol::Version version,
            const QSslConfiguration* sslConfig,
            QObject* parent = NULL);
#endif // QT_NO_SSL
    Network(const QString& origin,
            QWebSocketProtocol::Version version,
            QObject* parent = NULL);
#endif // QT_WEBSOCKETS_LIB
    Network(SocketInterface* socketInterface, TimerInterface* timerInterface,
            QObject* parent = NULL);
    ~Network();

    void sendFrame(const Frame &frame);
    bool isConnectedToHost() const;
    bool autoReconnect() const;
    void setAutoReconnect(const bool autoReconnect);
    QAbstractSocket::SocketState state() const;
    int autoReconnectInterval() const;
    void setAutoReconnectInterval(const int autoReconnectInterval);
#ifndef QT_NO_SSL
    void ignoreSslErrors(const QList<QSslError>& errors);
    QSslConfiguration sslConfiguration() const;
    void setSslConfiguration(const QSslConfiguration& config);
#endif // QT_NO_SSL

public slots:
    void connectToHost(const QHostAddress& host, const quint16 port);
    void connectToHost(const QString& hostName, const quint16 port);
    void disconnectFromHost();
#ifndef QT_NO_SSL
    void ignoreSslErrors();
#endif // QT_NO_SSL

protected slots:
    void onSocketError(QAbstractSocket::SocketError socketError);

protected:
    void initialize();

    quint16 _port;
    QHostAddress _host;
    QString _hostName;
    bool _autoReconnect;
    int _autoReconnectInterval;
    SocketInterface* _socket;
    TimerInterface* _autoReconnectTimer;

    enum ReadState {
        Header,
        Length,
        PayLoad
    };

    ReadState _readState;
    quint8 _header;
    int _length;
    int _shift;
    QByteArray _data;

protected slots:
    void onSocketReadReady();
    void onDisconnected();
    void connectToHost();

private:
    Q_DISABLE_COPY(Network)
};

} // namespace QMQTT

#endif // QMQTT_NETWORK_P_H
