/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef QMQTT_CLIENT_P_H
#define QMQTT_CLIENT_P_H

#include "qmqtt_client.h"

#include <QHostAddress>
#include <QString>
#include <QByteArray>
#include <QHash>
#include <QTimer>
#include <QAbstractSocket>

#ifdef QT_WEBSOCKETS_LIB
#include <QWebSocket>
#endif // QT_WEBSOCKETS_LIB

#ifndef QT_NO_SSL
#include <QSslConfiguration>
QT_FORWARD_DECLARE_CLASS(QSslError)
#endif // QT_NO_SSL

namespace QMQTT {

class NetworkInterface;

class ClientPrivate
{
public:
    ClientPrivate(Client* qq_ptr);
    ~ClientPrivate();

    /// Deleted copy-constructor
    ClientPrivate(const ClientPrivate&) = delete;
    /// Deleted assignment operator
    ClientPrivate& operator=(const ClientPrivate&) = delete;
    /// Deleted move-constructor
    ClientPrivate(const ClientPrivate&&) = delete;
    /// Deleted move-operator
    ClientPrivate& operator=(const ClientPrivate&&) = delete;

    void init(const QHostAddress& host, const quint16 port, NetworkInterface* network = NULL);
#ifndef QT_NO_SSL
    void init(const QString& hostName, const quint16 port, const QSslConfiguration& config,
              const bool ignoreSelfSigned=false);
#endif // QT_NO_SSL
    void init(const QString& hostName, const quint16 port, const bool ssl, const bool ignoreSelfSigned);
#ifdef QT_WEBSOCKETS_LIB
#ifndef QT_NO_SSL
    void init(const QString& url,
              const QString& origin,
              QWebSocketProtocol::Version version,
              const QSslConfiguration* sslConfig,
              bool ignoreSelfSigned);
#endif // QT_NO_SSL
    void init(const QString& url,
              const QString& origin,
              QWebSocketProtocol::Version version);
#endif // QT_WEBSOCKETS_LIB
    void init(NetworkInterface* network);

    QHostAddress _host;
    QString _hostName;
    quint16 _port;
#ifdef QT_WEBSOCKETS_LIB
    QWebSocketProtocol::Version _webSocketVersion;
#endif // QT_WEBSOCKETS_LIB
#ifndef QT_NO_SSL
    bool _ignoreSelfSigned;
#endif // QT_NO_SSL
    quint16 _gmid;
    MQTTVersion _version;
    QString _clientId;
    QString _username;
    QByteArray _password;
    bool _cleanSession;
    ConnectionState _connectionState;
    QScopedPointer<NetworkInterface> _network;
    QTimer _timer;
    QTimer _pingResponseTimer;
    QString _willTopic;
    quint8 _willQos;
    bool _willRetain;
    QByteArray _willMessage;
    QHash<quint16, QString> _midToTopic;
    QHash<quint16, Message> _midToMessage;

    Client* const q_ptr;

    quint16 nextmid();
    void connectToHost();
    void sendConnect();
    void onTimerPingReq();
    void onPingTimeout();
    quint16 sendUnsubscribe(const QString &topic);
    quint16 sendSubscribe(const QString &topic, const quint8 qos);
    quint16 sendPublish(const Message &message);
    void sendPuback(const quint8 type, const quint16 mid);
    void sendDisconnect();
    void sendFrame(const Frame &frame);
    void disconnectFromHost();
    void stopKeepAlive();
    void onNetworkConnected();
    void onNetworkDisconnected();
    quint16 publish(const Message& message);
    void puback(const quint8 type, const quint16 msgid);
    void subscribe(const QString& topic, const quint8 qos);
    void unsubscribe(const QString& topic);
    void onNetworkReceived(const QMQTT::Frame& frame);
    void handleConnack(const quint8 ack);
    void handlePublish(const Message& message);
    void handlePuback(const quint8 type, const quint16 msgid);
    void handleSuback(const QString& topic, const quint8 qos);
    void handleUnsuback(const QString& topic);
    void handlePingresp();
    bool autoReconnect() const;
    void setAutoReconnect(const bool autoReconnect);
    int autoReconnectInterval() const;
    void setAutoReconnectInterval(const int autoReconnectInterval);
    bool isConnectedToHost() const;
    QMQTT::ConnectionState connectionState() const;
    void setCleanSession(const bool cleanSession);
    bool cleanSession() const;
    void setKeepAlive(const quint16 keepAlive);
    quint16 keepAlive() const;
    void setPassword(const QByteArray& password);
    QByteArray password() const;
    void setUsername(const QString& username);
    QString username() const;
    void setVersion(const MQTTVersion);
    MQTTVersion version() const;
    void setClientId(const QString& clientId);
    QString clientId() const;
    void setPort(const quint16 port);
    quint16 port() const;
    void setHost(const QHostAddress& host);
    QHostAddress host() const;
    void setHostName(const QString& hostName);
    QString hostName() const;
    void setWillTopic(const QString& willTopic);
    void setWillQos(const quint8 willQos);
    void setWillRetain(const bool willRetain);
    void setWillMessage(const QByteArray& willMessage);
    QString willTopic() const;
    quint8 willQos() const;
    bool willRetain() const;
    QByteArray willMessage() const;
    void onNetworkError(QAbstractSocket::SocketError error);
#ifndef QT_NO_SSL
    void ignoreSslErrors();
    void ignoreSslErrors(const QList<QSslError>& errors);
    QSslConfiguration sslConfiguration() const;
    void setSslConfiguration(const QSslConfiguration& config);
    void onSslErrors(const QList<QSslError>& errors);
#endif // QT_NO_SSL

    Q_DECLARE_PUBLIC(Client)
};

} // namespace QMQTT

#endif // QMQTT_CLIENT_P_H
