/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_BLUETOOTH_DEVICEHANDLER_H
#define OSDEV_COMPONENTS_BLUETOOTH_DEVICEHANDLER_H

#include "bluetoothbaseclass.h"
#include "deviceinfo.h"

#include <QDateTime>
#include <QTimer>
#include <QVector>

#include <QLowEnergyController>
#include <QLowEnergyService>

namespace osdev {
namespace components {
namespace bluetooth {

class DeviceHandler : public BluetoothBaseClass
{
    Q_OBJECT

public:
    enum class AddressType
    {
        PublicAddress,
        RandomAddress
    };

    DeviceHandler( QObject *parent = nullptr );

    /// Deleted copy-constructor
    DeviceHandler( const DeviceHandler& ) = delete;
    /// Deleted assignment operator
    DeviceHandler& operator=(const DeviceHandler&) = delete;
    /// Deleted move-constructor
    DeviceHandler( DeviceHandler&& ) = delete;
    /// Deleted move operator
    DeviceHandler& operator=( DeviceHandler&& ) = delete;

    void        setDevice( DeviceInfo *device );
    void        setAddressType( AddressType type );
    AddressType addressType() const;

    bool        measuring() const;
    bool        alive() const;

    void        setRequestedServiceUuid( const QBluetoothUuid &service_uuid );
    QBluetoothUuid requestedServiceUuid() const;

    void        setRequestedCharacteristicUuid( const QBluetoothUuid &char_uuid );
    QBluetoothUuid requestedCharacteristicUuid() const;

signals:
    void        signalMeasuringChanged();
    void        signalAliveChanged();
    void        signalConnectionStateChanged( bool state );
    void        signalStatsChanged();

    void        signalReceivedValue( const QString &value );

public slots:
    void        slotStartMeasurement();
    void        slotStopMeasurement();
    void        slotDisconnectService();

    // QLowEnergyController
    void        slotServiceDiscovered( const QBluetoothUuid &gatt );
    void        slotServiceScanDone();

    // QLowEnergyService
    void        slotServiceStateChanged( QLowEnergyService::ServiceState state );
    void        slotUpdateMeasuredValue( const QLowEnergyCharacteristic &characteristic, const QByteArray &value );
    void        slotConfirmedDescriptorWrite( const QLowEnergyDescriptor &descriptor, const QByteArray &value );

private:
    QLowEnergyController    *m_pControl;
    QLowEnergyService       *m_pService;
    QLowEnergyDescriptor     m_notificationDescriptor;
    DeviceInfo              *m_pCurrentDevice;

    bool                     m_foundRequestedService;
    bool                     m_measuring;

    QVector<int>             m_measurements;
    QLowEnergyController::RemoteAddressType m_addressType;
    QBluetoothUuid           m_requestedServiceUuid;
    QBluetoothUuid           m_requestedCharacteristicUuid;
};

}       /* End namespace bluetooth  */
}       /* End namespace components */
}       /* End namespace osdev      */

#endif  /* OSDEV_COMPONENTS_BLUETOOTH_DEVICEHANDLER_H */
