/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_TIMESTAMP_H
#define OSDEV_COMPONENTS_TIMESTAMP_H

#include <limits>
#include <stdexcept>

#include <QString>

namespace osdev {
namespace components {

/**
 * @brief Timestamp class that allows formation of a sequence of ordered events.
 */
class Timestamp
{
public:
    /**
     * @brief Creates an invalid timestamp. This is the oldest possible timestamp.
     * @note A timestamp can only become valid by assigning a valid timestamp to it.
     */
    Timestamp();

    /**
     * @brief Creates a timestamp based on miliseconds since the epoch.
     * @param msecsSinceEpoch Number of miliseconds since the epoch. Value 0 leads to valid timestamp.
     * @note Timestamp with value 0 is larger then the invalid timestamp.
     */
    explicit Timestamp(unsigned long long msecsSinceEpoch);


    // default constructable and movable.
    Timestamp(const Timestamp&) = default;
    Timestamp& operator=(const Timestamp&) = default;
    Timestamp(Timestamp&&) = default;
    Timestamp& operator=(Timestamp&&) = default;

    /**
     * @return true if this timestamp is valid, false otherwise.
     */
    bool valid() const
    {
        return (m_seqNr > 0);
    }

    /**
     * @return true if this timestamp is smaller (older) then rhs.
     */
    bool operator<(const Timestamp& rhs) const;

    /**
     * @brief Prefix addition operator.
     * Increases the Timestamp sequencenumber so that timestamps with the same number of msecs since the epoch can also be ordered.
     * @note An invalid timestamp remains invalid.
     * @return Reference to this timestamp.
     * @throw std::out_of_range exception when seqnr is about to overflow.
     */
    Timestamp& operator++();

    /**
     * @brief Postfix addition operator.
     * Increases the Timestamp sequencenumber so that timestamps with the same number of msecs since the epoch can also be ordered.
     * @note An invalid timestamp remains invalid.
     * @return The timestamp as it was before increment.
     * @throw std::out_of_range exception when seqnr is about to overflow.
     */
    Timestamp operator++(int);

    /**
     * @return Stringified timestamp as msecsSinceEpoch:seqNr.
     */
    QString toString() const;

private:
    unsigned long long m_msecsSinceEpoch;   ///< Number of miliseconds since the epoch.
    unsigned long long m_seqNr;             ///< Sequence number.
};

}       /* End namespace components */
}       /* End namespace osdev      */

#endif  /* OSDEV_COMPONENTS_TIMESTAMP_H */
