/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include "timestamp.h"

using namespace osdev::components;

Timestamp::Timestamp()
    : m_msecsSinceEpoch(0)
    , m_seqNr(0)
{
}

Timestamp::Timestamp(unsigned long long msecsSinceEpoch)
    : m_msecsSinceEpoch(msecsSinceEpoch)
    , m_seqNr(1)
{
}

bool Timestamp::operator<(const Timestamp& rhs) const
{
    if (m_msecsSinceEpoch < rhs.m_msecsSinceEpoch)
    {
        return true;
    }

    if (m_msecsSinceEpoch == rhs.m_msecsSinceEpoch)
    {
        return m_seqNr < rhs.m_seqNr;
    }
    return false;
}

Timestamp& Timestamp::operator++() // prefix
{
    if (!valid())
    {
        return *this; // an invalid timestamp can only be made valid by assigning a valid timestamp to it.
    }

    if (m_seqNr < std::numeric_limits<decltype(m_seqNr)>::max())
    {
        ++m_seqNr;
    }
    else
    {
        // The idea is to never reach this point by choosing a large datatype.
        // Overflowing is not an option because then the < relation doesn't hold anymore.
        throw std::out_of_range("sequence number overflow");
    }
    return *this;
}

Timestamp Timestamp::operator++(int) // postfix
{
    auto cpy(*this);
    ++(*this);
    return cpy;
}

QString Timestamp::toString() const
{
    return QString("%1:%2").arg(m_msecsSinceEpoch).arg(m_seqNr);
}

