/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_TIMELINE_H
#define OSDEV_COMPONENTS_TIMELINE_H

// boost
#include <boost/circular_buffer.hpp>

#include "ormbatchchange.h"

namespace osdev {
namespace components {

/**
 * @brief The timeline contains the changes that where made to a specific Batch/Merge update table.
 * Only a small number of changes is recorded. A change arrives that is older then the oldest recorded
 * change this change will not be considered.
 */
class Timeline
{
public:
    /**
     * @brief Creates an empty timeline.
     */
    Timeline();

    // Non copyable, non movable.
    Timeline(const Timeline&) = delete;
    Timeline& operator=(const Timeline&) = delete;
    Timeline(Timeline&&) = delete;
    Timeline& operator=(Timeline&&) = delete;

    /**
     * @brief Evaluate an incoming change against the already registered changes.
     * A previously evaluated change is discarded.
     * @param desiredChange The incoming change.
     * @param exclusiveUpdate Flag that indicates if this change should be evaluated in an exclusive manner.
     * @return The reshaped and possibly invalidated change that is left.
     * @note The evaluated change is cached so that it can be commited if the database update actually succeeds.
     */
    const OrmBatchChange& evaluate(const OrmBatchChange& desiredChange, bool exclusiveUpdate);

    /**
     * @brief Commit an evaluated change.
     * Only valid proposals are commited. This method should be called when the change
     * is sucessfuly written to the database.
     * @note To make this somewhat trustworthy the underlying database should support transactions.
     */
    void commit();

private:
    boost::circular_buffer<OrmBatchChange> m_timeline;  ///< Only a small number of changes are buffered.
    OrmBatchChange m_proposedChange;                    ///< Cache the evaluated proposed change for commiting.
};

}   /* End namespace components */
}   /* End namespace osdev */

#endif  /* OSDEV_COMPONENTS_TIMELINE_H */
