/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_ORMTHREAD_H
#define OSDEV_COMPONENTS_ORMTHREAD_H

#include <QThread>
#include <QSharedPointer>

#include "ormreldata.h"

/*  ______________________________________
 * / Good day to deal with people in high \
 * | places; particularly lonely          |
 * \ stewardesses.                        /
 *  --------------------------------------
 *    \
 *     \
 *         .--.
 *        |o_o |
 *        |:_/ |
 *       //   \ \
 *      (|     | )
 *     /'\_   _/`\
 *     \___)=(___/
 *//*!
 *  \brief  This class *manages* the thread, the ORM layer will be started in.
 *          Derived from QThread it isn't the thread itself but implements the
 *          'run()' method that'll start the thread. (QThread isn't meant to be
 *          instantiated. It has to be derived.. :) )
 *
 *          And sometimes it can be a Stewardess. That's polymorphism for yah!
 */
namespace osdev  {
namespace components {

class ORMThread : public QThread
{
    Q_OBJECT

public:
    ORMThread();

    // Deleted copy-constructor
    ORMThread( const ORMThread& ) = delete;
    ORMThread( const ORMThread&& ) = delete;
    ORMThread& operator=( const ORMThread& ) = delete;
    ORMThread& operator=( const ORMThread&& ) = delete;

    /*!
     * \brief Destructor
     */
    virtual ~ORMThread();

    /*!
     * \brief   Override from QThread. This will actually create and start the thread.
     *          Objects used here are "moved" to the new thread and run
     */
    void run() override;

    /*!
     * \brief   Send the data to the intended thread.
     * \param   data Data coming from the plugin. Ownership is moved to the thread and that
     *          specific thread is responsible for cleaning up.
     */
    void dataToThread( const QSharedPointer<ORMRelData>& data );

signals:
    /*!
     * \brief   Used to send the datastructure to the thread.
     * \param   data Data coming from the plugin.
     */
    void signalSendData( const QSharedPointer<ORMRelData>& data );

    /*!
     * \brief   If the thread is unable to process the data that was sent, it will "reject" this.
     *          Normally it would then be send to a subsystem, capable of storing the data and
     *          resend it for re-processing.
     * \param   data Data originally comming from the plugin (it can be enriched along the way).
     */
    void signalRejectedData( const QSharedPointer<ORMRelData>& data );

};


}   /* End namespace components */
}   /* End namespace osdev  */

#endif  /* OSDEV_COMPONENTS_ORMTHREAD_H */
