/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_ORMHANDLER_H
#define OSDEV_COMPONENTS_ORMHANDLER_H

#include <memory>

#include <QObject>
#include <QHash>
#include <QSharedPointer>
#include <QString>

#include "dbconnector.h"
#include "ormtable.h"
#include "ormreldata.h"

namespace osdev  {
namespace components {

/**
 * @brief Handles the Object Relational Model
 */
class OrmHandler : public QObject
{
    Q_OBJECT

public:
    enum class OrmStatus
    {
        OrmStatusSuccess    = 0,
        OrmStatusNoRelation,
        OrmStatusFailed
    };

    /**
     * @brief Constructor
     * @param parent Parent QObject
     */
    explicit OrmHandler(QObject *parent = nullptr);

    /// Deleted copy-constructor
    OrmHandler(const OrmHandler&) = delete;
    /// Deleted assignment operator
    OrmHandler& operator=(const OrmHandler&) = delete;
    /// Deleted move-constructor
    OrmHandler(const OrmHandler&&) = delete;
    /// Deleted move operator
    OrmHandler& operator=(const OrmHandler&&) = delete;

    virtual ~OrmHandler();

    /// @brief Start the ORM handler
    void start();

public slots:
    /**
     * @brief   Receive data from an external source
     * @param   dataContainer  - Data received Each layer has shared ownership of the
     *          pointer it receives. (See SSD @TODO Actualy SSD needs updating on this issue.)
     */
    void receiveData( const QSharedPointer<ORMRelData>& dataContainer );

    /*!
     * \brief   Gets the primary key from the given tableName
     * \param   tableName - The table we're interested in.
     * \return  The primary keyfield name.
     */
    QString getPrimaryKey( const QString& tableName ) const;

signals:
    /*!
     * \brief Signal for RejectedData.
     * \param dataContainer The data container for this signal.
     */
    void signalRejectedData( const QSharedPointer<ORMRelData>& dataContainer );

private:
    /**
     * @brief Create an Object Relational Model object for a specific table
     * @param _table Table name
     * @return True if successful, false otherwise
     */
    bool createOrmObject( const QString& _table );

    /*!
     * \brief   Create an ORM object for a specific relation or constraint.
     *
     * \param   table The name of the table for which to create an ORM relation.
     * \return  true if the object was created. False if something went wrong.
     */
    OrmStatus createOrmRelation( const QString& table );

    // Internal structures and objects.
    QHash<QString, OrmTable*> m_qhOrmTables;    ///< Maps table-name to TableModel*
    DbConnector m_dbConnection;                 ///< Database connection
    QString m_className;                        ///< Name of the class for logging purposes
};

} // End namespace components
} // End namespace osdev

#endif // OSDEV_COMPONENTS_ORMHANDLER_H
