/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

#ifndef OSDEV_COMPONENTS_CTCPSOCKET_H
#define OSDEV_COMPONENTS_CTCPSOCKET_H

#include <QObject>
//#include <QTimer>
//#include <QTcpSocket>
//#include <QHostAddress>
//#include <QDataStream>
//#include <QString>

#include <memory>

class QTcpSocket;
class QTimer;

namespace osdev  {
namespace components {

/**
 * @brief Wrapper around QTcpSocket
 */
class TcpSocket : public QObject
{
    Q_OBJECT

public:
    /**
     * @brief Create a TcpSocket connecting to the given address
     * @param ipAddress Host to connect to
     * @param portNumber Port number to connect at
     */
    TcpSocket( const QString& ipAddress, int portNumber );

    /**
     * @brief Create a TcpSocket based on an existing QTcpSocket
     * @param pSocket Socket used. This class does NOT take ownership of the
     *                provided socket.
     */
    TcpSocket( QTcpSocket *pSocket );

    /// Deleted copy-constructor
    TcpSocket(const TcpSocket&) = delete;
    /// Deleted assignment operator
    TcpSocket& operator=(const TcpSocket&) = delete;
    /// Deleted move-constructor
    TcpSocket(TcpSocket&&) = delete;
    /// Deleted move operator
    TcpSocket& operator=(TcpSocket&&) = delete;

    /// @brief Destructor
    ~TcpSocket();

    /**
     * @brief Retrieve the last error registered in the socket
     * @return Error string
     */
    QString     showError();

    /**
     * @brief See if a connection is available
     * @return True if a connection is available, false otherwise
     */
    bool        isConnected();

public slots:
    /**
     * @brief Slot called to send data
     * @param sData Data to send
     * @param pSocket Socket to which to send. If it is not equal to this, the
     *                slot does nothing and returns
     */
    void        slotSendData( const QString &sData, TcpSocket *pSocket );

    /**
     * @brief Updates the current QTcpSocket used
     * @param pSocket New socket
     */
    void        slotSetSocket( QTcpSocket *pSocket );

    /// @brief Slot called when a socket gets connected
    void        slotConnected();

    /// @brief Slot called when a socket gets disconnected
    void        slotDisconnected();

signals:
    /**
     * @brief Signal emitted when data is received
     * @param sData Block of data received
     * @param pSocket Receiving socket
     */
    void        signalDataReceived( const QString &sData, TcpSocket *pSocket );

    /**
     * @brief Signal emitted when data was sent
     * @param pSocket Sending socket
     */
    void        signalDataSent( TcpSocket *pSocket );

    /**
     * @brief Signal emitted when data is being received
     * @param pSocket Receiving socket
     */
    void        signalReceivingData( TcpSocket *pSocket );

    /**
     * @brief Signal emitted when a socket gets connected
     * @param pSocket Socket that gets connected
     */
    void        signalConnected( TcpSocket *pSocket );

    /**
     * @brief Signal emitted when a socket gets disconnected
     * @param pSocket Socket that gets disconnected
     */
    void        signalDisconnected( TcpSocket *pSocket );

private slots:
    /// @brief Slot called when a socket is ready to read
    void        slotReadyRead();

    /// @brief Emits the signalDataSent when all data was sent
    void        chkSendBuffer();

private:
    /// @brief Connects the socket signals to the slots of this class
    void        connectSocketSignals();

    /// @brief Disconnects the socket signals to the slots of this class
    void        disconnectSocketSignals();

    QTcpSocket              *m_pSocket;     ///< Current communication socket
    int                     m_blockSize;    ///< Maximum block size
    QString                 m_dataBuffer;   ///< Data to be sent

    std::unique_ptr<QTimer> m_pTimer;       ///< Periodic check if all data was sent
};

} // End namespace components
} // End namespace osdev 

#endif // OSDEV_COMPONENTS_CTCPSOCKET_H
