/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

#ifndef OSDEV_COMPONENTS_CTCPINTERFACE_H
#define OSDEV_COMPONENTS_CTCPINTERFACE_H

#include <QObject>
#include <QList>

class QTcpServer;
class QTcpSocket;

namespace osdev  {
namespace components {

/**
 * @brief Handles a generic client or server point-to-point connection
 *
 * This class can be instantiated as a client or a server.
 *
 * @note The server can currently only handle one client at a time.
 */
class TcpInterface : public QObject
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     * @param hostName Host to connect with (client-mode), or interface to
     *                 connect at (server-mode)
     * @param port Selected port-number
     * @param i_bServer False for client-mode, true for server-mode
     */
    TcpInterface(const QString& hostName, int port = 2776, bool i_bServer = false);
    /// @brief Destructor
    ~TcpInterface();

    /// Deleted copy-constructor
    TcpInterface(const TcpInterface&) = delete;
    /// Deleted assignment operator
    TcpInterface& operator=(const TcpInterface&) = delete;
    /// Deleted move-constructor
    TcpInterface(TcpInterface&&) = delete;
    /// Deleted move operator
    TcpInterface& operator=(TcpInterface&&) = delete;

    /**
     * @return if the interface is active.
     * In server mode the interface is listening or connected to and in
     * client mode the interface is connected.
     */
    bool active() const
    {
        return m_active;
    }

    /**
     * @brief Reads the first block of data from the received data
     * @return First block of data, as a QString
     */
    QString getData();

public slots:

    /**
     * @brief Called when data is ready to be read into the buffer
     */
    void readData();

    /**
     * @brief Send a block of data to the other party
     * @param i_qsData Data to be sent
     */
    void sendData(const QString& i_qsData);

signals:
    /// @brief Emitted when data is present on the connection
    void dataPresent();

private slots:
    /// @brief Slot called when a new connection is made
    void newConnection();

private:
    bool m_bServer;                 ///< True for server-mode, false for client-mode
    bool m_active;                  ///< True if in server mode the server is listening or connected to and in client mode when the client is connected.

    QTcpServer* m_tcpServer;        ///< Server object instance
    QTcpSocket* m_tcpSocket;        ///< Client object instance

    QTcpSocket* m_clientConnection; ///< Current client connection

    quint32 m_blockSize;            ///< Maximum size of a data-block
    QList<QByteArray> m_dataList;   ///< Blocks of data read from the counterpart
};

} // End namespace components
} // End namespace osdev 

#endif /* OSDEV_COMPONENTS_CTCPINTERFACE_H */
