/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

#ifndef OSDEV_COMPONENTS_CTHREADCONTAINER_H
#define OSDEV_COMPONENTS_CTHREADCONTAINER_H

/***************************************************************************
 * Global Includes
 ***************************************************************************/
#include <QHash>

/***************************************************************************
 * Local Includes
 ***************************************************************************/

namespace osdev  {
namespace components {

class TcpSocket;

/**
 * @brief Maintains a collection of sockets
 *
 * @note This class does NOT take ownership of the provided sockets
 */
class SocketContainer
{
public:
    /// Constructor
    SocketContainer();
    /// Destructor
    ~SocketContainer();

    /// Deleted copy-constructor
    SocketContainer(const SocketContainer&) = delete;
    /// Deleted assignment operator
    SocketContainer& operator=(const SocketContainer&) = delete;
    /// Deleted move-constructor
    SocketContainer(SocketContainer&&) = delete;
    /// Deleted move operator
    SocketContainer& operator=(SocketContainer&&) = delete;

    //! This function will set the given thread to free
    //! \param pSocket Socket to set to free
    void setChannelFree( TcpSocket *pSocket );

    //! This function will set the given thread to busy
    //! \param pSocket Socket to set to busy
    void setChannelBusy( TcpSocket *pSocket );

    //! This function will add a newly created thread to the container
    //! \param pSocket Add a new socket to the list
    void addSocket( TcpSocket *pSocket );

    //! This function removes a thread from the list for whatever reason there might be
    //! \param pSocket Removes the socket from the list
    void removeSocket( TcpSocket *pSocket );

    //! @return This function will return the first free thread.
    TcpSocket* getFreeSocket() const;

private:
    /// Dump the status to debug log
    void printStatus() const;

    /// Contains all managed sockets
    QHash< TcpSocket*, bool > m_hshFreeBySocket;
};

} // End namespace components
} // End namespace osdev 

#endif /* OSDEV_COMPONENTS_CTHREADCONTAINER_H */
