/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include "msgparser.h"

namespace osdev  {
namespace components {

const QString MsgParser::s_messageElementTag = "message";
const QString MsgParser::s_sourceElementTag = "source";
const QString MsgParser::s_destElementTag = "dest";
const QString MsgParser::s_actionElementTag = "action";
const QString MsgParser::s_dataElementTag = "data";
const QString MsgParser::s_idAttributeTag = "id";

MsgParser::MsgParser(QObject* _parent)
    : QObject(_parent)
    , m_message()
    , m_sourceId()
    , m_destId()
    , m_actionId(-1)
    , m_actionData()
{
}

MsgParser::~MsgParser()
{
}

void MsgParser::createMsg(const QString& sourceId, const QString& destId,
                          int actionId, const QString& actionData)
{
    static const QString xmlVar("<%1 %2='%3'/>");
    static const QString xmlData("<%1>%2</%1>");
    static const QString xmlMessage("<%1>%2 %3 %4 %5</%1>");

    QString src = xmlVar.arg(s_sourceElementTag).arg(s_idAttributeTag).arg(sourceId);
    QString dst = xmlVar.arg(s_destElementTag).arg(s_idAttributeTag).arg(destId);
    QString action = xmlVar.arg(s_actionElementTag).arg(s_idAttributeTag).arg(actionId);
    QString data = xmlData.arg(s_dataElementTag).arg(actionData);

    m_message = xmlMessage.arg(s_messageElementTag).arg(src).arg(dst).arg(action).arg(data);
}

void MsgParser::createMsg(const QString& sourceId, const QString& destId,
                          int actionId, xercesc_2_7::DOMDocument* actionData)
{
    QString msgData;

    CXmlParser* pSerializer = new CXmlParser();
    // Store the complete message.
    msgData = pSerializer->serialize(actionData);
    delete pSerializer;

    static const QString xmlVar("<%1 %2='%3'/>");
    static const QString xmlData("<%1>%2</%1>");
    static const QString xmlMessage("<%1>%2 %3 %4 %5</%1>");

    QString src = xmlVar.arg(s_sourceElementTag).arg(s_idAttributeTag).arg(sourceId);
    QString dst = xmlVar.arg(s_destElementTag).arg(s_idAttributeTag).arg(destId);
    QString action = xmlVar.arg(s_actionElementTag).arg(s_idAttributeTag).arg(actionId);
    QString data = xmlData.arg(s_dataElementTag).arg(msgData);

    m_message = xmlMessage.arg(s_messageElementTag).arg(src).arg(dst).arg(action).arg(data);
}

void MsgParser::parseMsg(const QString& msg)
{
    CXmlParser parser;

    DOMDocument* domDocument = parser.parse(msg);
    DOMNode* rootNode = domDocument->getDocumentElement();

    DOMNode* node =  rootNode->getFirstChild();
    if (CXerces::nodeName(node) == sourceElementTag)
    {
        m_sourceId = CXerces::getNodeAttribute(node, qPrintable(s_idAttributeTag));
    }

    node = node->getNextSibling();
    if (CXerces::nodeName(node) == destElementTag)
    {
        m_destId = CXerces::getNodeAttribute(node, qPrintable(s_idAttributeTag));
    }

    node = node->getNextSibling();
    if (CXerces::nodeName(node) == s_actionElementTag)
    {
        m_actionId = CXerces::getNodeAttribute(node, qPrintable(s_idAttributeTag)).toInt();
    }

    node = node->getNextSibling();
    if (CXerces::nodeName(node) == s_dataElementTag)
    {
        m_actionData = parser.serialize(node);
    }
}

QString MsgParser::getMsg() const
{
    return m_message;
}

QString MsgParser::getSourceId() const
{
    return m_sourceId;
}

QString MsgParser::getDestId() const
{
    return m_destId;
}

int MsgParser::getActionId() const
{
    return m_actionId;
}

QString MsgParser::getActionData() const
{
    return m_actionData;
}

}
}
