/* ****************************************************************************
 * Copyright 2024 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include <iostream>

#include "argumentparser.h"
#include "filereader.h"
#include "publisher.h"

enum TIME_RES
{
    T_MICRO,
    T_MILLI,
    T_SECONDS
};

std::uint64_t getEpochUSecs()
{
    auto tsUSec =std::chrono::time_point_cast<std::chrono::microseconds>(std::chrono::system_clock::now());
    return static_cast<std::uint64_t>(tsUSec.time_since_epoch().count());
}


void sleepcp( int number, TIME_RES resolution = T_MILLI )    // Cross-platform sleep function
{
    int factor = 0; // Should not happen..

    switch( resolution )
    {
        case T_MICRO:
            factor = 1;
            break;

        case T_MILLI:
            factor = 1000;
            break;

        case T_SECONDS:
            factor = 1000000;
        break;
    }

    usleep( number * factor );
}

int main(int argc, char* argv[])
{
    ArgumentParser parser(argc, argv);

    std::string host = parser.getHost();
    std::string port = parser.getPort();
    std::string user = parser.getUserName();
    std::string pass = parser.getPassword();
    std::string topic = parser.getTopic();
    std::string file = parser.getFile();
    std::string message = parser.getMessage();

    if(!file.empty() && !message.empty())
    {
        std::cerr << "Error: Cannot specify both file and message" << std::endl;
        return -1;
    }

    if(file.empty() && message.empty())
    {
        std::cerr << "Error: Must specify either file or message" << std::endl;
        return -1;
    }

    std::string content = std::string();
    if(!file.empty())
    {
        content = FileReader::read_file(file);
        if(content.empty())
        {
            std::cerr << "Error: File is empty" << std::endl;
            return -1;
        }
    }

    if(!message.empty())
    {
        content = message;
    }

    // Ok, we have all the options, lets check the mandatory ones
    if(topic.empty())
    {
        std::cerr << "Error: Must specify a topic" << std::endl;
        return -1;
    }

    // Create the MQTT client
    Publisher oPublisher;
    oPublisher.connect(host, std::stoi(port), user, pass);

    // Wait a couple of seconds to make sure the connection is established
    sleepcp(2, T_SECONDS);

    oPublisher.publish(topic, content);
    sleepcp(2, T_SECONDS);

    return 0;
}