/****************************************************************************
 * COpyright (c) 2023 Open Systems Development B.V.
 ****************************************************************************/

#include <gmock/gmock.h>
#include <gtest/gtest.h>
#include <string>
#include <memory>
#include <chrono>
#include <unistd.h>

#include <iostream>

#include "helperclasses/PublisherClass.h"

/// Every test does basically the same.
/// 1. Create a Publisher object
/// 2. Connect to the MQTT broker
/// 3. Publish 10 message(s)
/// 4. Disconnect from the MQTT broker
/// 5. Destroy the Publisher object
/// 6. Repeat 10 times.

const std::string sledge_maintopic = "SledgeHammerTest/";

enum TIME_RES
{
    T_MICRO,
    T_MILLI,
    T_SECONDS
};

std::uint64_t getEpochUSecs()
{
    auto tsUSec =std::chrono::time_point_cast<std::chrono::microseconds>(std::chrono::system_clock::now());
    return static_cast<std::uint64_t>(tsUSec.time_since_epoch().count());
}


void sleepcp( int number, TIME_RES resolution = T_MILLI )    // Cross-platform sleep function
{
    int factor = 0; // Should not happen..

    switch( resolution )
    {
        case T_MICRO:
            factor = 1;
            break;

        case T_MILLI:
            factor = 1000;
            break;

        case T_SECONDS:
            factor = 1000000;
        break;
    }

    usleep( number * factor );
}

/// Test a single connection, 1 time
TEST(SledgeHammerTest, SingleConnectionCleanExit)
{
    Publisher *publisher = nullptr;
    for (int test_counter = 0; test_counter < 10; test_counter++)
    {
        publisher = new Publisher(std::to_string(getEpochUSecs()));
        if(publisher)
        {
            publisher->connect("localhost");
            publisher->publish(sledge_maintopic + "Single Connection Clean Exit Test/" + std::to_string(test_counter), "Hello World. I'm alive..");
            publisher->disconnect();
            delete publisher;
            publisher = nullptr;
        }
    }
}

TEST(SledgeHammerTest, SingleConnectionForcedExit)
{
    Publisher *publisher = nullptr;
    for (int test_counter = 0; test_counter < 10; test_counter++)
    {
        publisher = new Publisher(std::to_string(getEpochUSecs()));
        if(publisher)
        {
            publisher->connect("localhost");
            publisher->publish(sledge_maintopic + "Single Connection Forced Exit/Test/" + std::to_string(test_counter), "Hello World. I'm alive..");
            delete publisher;
            publisher = nullptr;
        }
    }
}

TEST(SledgeHammerTest, MultipleConnections_10_CleanExit)
{
    std::unordered_map<std::string, Publisher *> publishers;

    for (int test_counter = 0; test_counter < 10; test_counter++)
    {
        publishers["Publisher" + std::to_string(test_counter)] = new Publisher(std::to_string(getEpochUSecs()));
        if(publishers["Publisher" + std::to_string(test_counter)])
        {
            publishers["Publisher" + std::to_string(test_counter)]->connect("localhost");
        }
    }

    for (int test_counter = 0; test_counter < 10; test_counter++)
    {
        for (int nCount = 0; nCount < 10; nCount++)
        {
            publishers["Publisher" + std::to_string(test_counter)]->publish(sledge_maintopic + "Multiple Connections [10]/Clean Exit/Test " + std::to_string(test_counter), "Hello World. I'm alive..");
        }
        publishers["Publisher" + std::to_string(test_counter)]->disconnect();
        delete publishers["Publisher" + std::to_string(test_counter)];
        publishers.erase("Publisher" + std::to_string(test_counter));
    }
}

TEST(SledgeHammerTest, MultipleConnections_BurnTest)
{
    std::unordered_map<std::string, Publisher *> publishers;

    const int max_run = 100;
    const int max_connections = 5;
    const int max_messages = 100;

    for (int test_run = 0; test_run < max_run; test_run++)
    {
        std::cout << "Creating " << max_connections << " connections for test run " << test_run << " " << std::endl;
        for (int test_counter = 0; test_counter < max_connections; test_counter++)
        {
            publishers["Publisher" + std::to_string(test_counter)] = new Publisher(std::to_string(getEpochUSecs()));
            if(publishers["Publisher" + std::to_string(test_counter)])
            {
                publishers["Publisher" + std::to_string(test_counter)]->connect("localhost");
                std::cout << ".";
            }
        }
        std::cout << std::endl;

        // Wait a second for the connections to be established.
        sleepcp(1, T_SECONDS);

        std::cout << "Publishing " << max_messages << " messages for test run " << test_run << std::endl;
        for (int test_counter = 0; test_counter < max_connections; test_counter++)
        {
            std::cout << "\tConnection : " << test_counter << " ";
            for (int nCount = 0; nCount < max_messages; nCount++)
            {
                if(publishers["Publisher" + std::to_string(test_counter)])
                {
                    std::cout << ".";
                    publishers["Publisher" + std::to_string(test_counter)]->publish(sledge_maintopic + 
                            "Multiple Connections [" + std::to_string(max_connections) + "]/Multi Messages [" + std::to_string(max_messages) + "]/Test Run " + 
                            std::to_string(test_run) + "/" + "[" + std::to_string(test_counter) + "]" +
                                                                "[" + std::to_string(nCount) + "]", "Hello World. I'm alive..");
                }
            }
            std::cout << std::endl;
        }

        // Wait another second to update the broker.
        sleepcp(1, T_SECONDS);

        std::cout << "Disconnecting " << max_connections << " connections for test run " << test_run << " ";
        for (int test_counter = 0; test_counter < max_connections; test_counter++)
        {
            if(publishers["Publisher" + std::to_string(test_counter)])
            {
                std::cout << ".";
                publishers["Publisher" + std::to_string(test_counter)]->disconnect();
                delete publishers["Publisher" + std::to_string(test_counter)];
            }
        }
        std::cout << std::endl;
        publishers.clear();
        std::cout << std::string(200,'=') << std::endl;

        // Wait for another second before re-running another 100 times.
        sleepcp(1, T_SECONDS);
    }
}