/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include "serverstate.h"

using namespace osdev::components::mqtt;

const std::string ServerState::s_identifier("ServerState");

std::atomic<StateChangeCallbackHandle> ServerState::s_nextServerStateCallbackHandle(1);

ServerState::ServerState(const IStateCallback* stateCallbackIf)
    : sig_serverStateChanged()
    , m_stateCallbackIf(stateCallbackIf)
    , m_serverStateCallbackMap()
    , m_state(StateEnum::Unknown)
{
}

ServerState::~ServerState()
{
    try {
        this->clearAllStateChangeCallbacks();
    }
    catch (...) {
    }
}

StateChangeCallbackHandle ServerState::registerStateChangeCallback(const IStateCallback::SlotStateChange& cb)
{
    const auto handle = s_nextServerStateCallbackHandle++;
    m_serverStateCallbackMap.emplace(std::piecewise_construct,
        std::make_tuple(handle),
        std::make_tuple(sig_serverStateChanged.connect(cb)));

    return handle;
}

void ServerState::unregisterStateChangeCallback(StateChangeCallbackHandle handle)
{
    m_serverStateCallbackMap.erase(handle);
}

void ServerState::clearAllStateChangeCallbacks()
{
    this->emitStateChanged(StateEnum::Unregister);
    m_serverStateCallbackMap.clear();
}

void ServerState::emitStateChanged(StateEnum newState)
{
    m_state = newState; ///< store state, to allow clients to retrieve the last received state.

    if (nullptr != m_stateCallbackIf) {
        sig_serverStateChanged(m_stateCallbackIf, newState);
    }
}

StateEnum ServerState::state() const
{
    return m_state;
}

std::string ServerState::stateCallbackClientId() const
{
    static const std::string nullId = "null";
    return (nullptr == m_stateCallbackIf) ? nullId : m_stateCallbackIf->clientId();
}
