/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include "mqtttypeconverter.h"

// std
#include <chrono>

// boost
#include <boost/lexical_cast.hpp>
#include <boost/uuid/uuid_io.hpp>

// date
#include "date.h"

#include "stringutils.h"

using namespace osdev::components::mqtt;

namespace osdev {
namespace components {
namespace mqtt {
namespace MqttTypeConverter {

std::string toStdString(const MqttId& mqttId)
{
    return boost::lexical_cast<std::string>(mqttId);
}

MqttId toMlogicId(const std::string& mqttId)
{
    return boost::lexical_cast<MqttId>(mqttId);
}

StdTime toStdTime(const time_t& posixTime)
{
    return std::chrono::system_clock::from_time_t(posixTime);
}

time_t toPosixTime(const StdTime& stdTime)
{
    return std::chrono::system_clock::to_time_t(stdTime);
}

OptionalTime toOptionalTime(const std::string& posixTimeString)
{
    static const OptionalTime nonExisting;

    try {
        auto lotPosixTime = boost::lexical_cast<std::time_t>(posixTimeString);
        return toStdTime(lotPosixTime);
    }
    catch (const boost::bad_lexical_cast&) {
        return nonExisting;
    }
}

std::string toShortGuidAppendedString(const std::string& str, const MqttId& mqttId)
{
    std::string idStr = toStdString(mqttId);
    return str + "-" + idStr.substr(0, 4);
}

}   // End namespace MqttTypeConverter
}   // End namespace mqtt
}   // End namespace components
}   // End namespace osdev
