/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include "mqttidgenerator.h"

// std
#include <chrono>
#include <mutex>
#include <thread>
#include <unistd.h>

// boost
#include <boost/uuid/uuid_generators.hpp>

#include "lockguard.h"
#include "commondefs.h"

namespace {

boost::uuids::basic_random_generator<boost::mt19937> createGenerator()
{
    static auto timeSeed = static_cast<boost::mt19937::result_type>(std::chrono::duration_cast<std::chrono::nanoseconds>(std::chrono::high_resolution_clock::now().time_since_epoch()).count());
    static auto pidSeed = static_cast<boost::mt19937::result_type>(getpid());
    static std::hash<std::thread::id> hasher;
    static auto threadSeed = static_cast<boost::mt19937::result_type>(hasher(std::this_thread::get_id()));
    static boost::mt19937 randomNumberGenerator;
    randomNumberGenerator.seed(timeSeed ^ pidSeed ^ threadSeed);
    return boost::uuids::basic_random_generator<boost::mt19937>(&randomNumberGenerator);
}

} // namespace

using namespace osdev::components::mqtt;

const MqttId mqttNamespace = boost::lexical_cast<boost::uuids::uuid>("9c0f3730-cc4f-49eb-ab5b-079bc5b0e481");

// static
MqttId MqttIdGenerator::generate()
{
    // From the boost design notes
    // Seeding is unqiue per random generator:
    // The boost::uuids::basic_random_generator class' default constructor seeds the random number generator
    //  with a SHA-1 hash of a number of different values including std::time(0), std::clock(), uninitialized data,
    //  value return from new unsigned int, etc..
    // Functions are reentrant:
    // All functions are re-entrant. Classes are as thread-safe as an int. That is an instance can not be shared
    //  between threads without proper synchronization.
    static auto uuidGenerator = createGenerator();
    static std::mutex generatorMutex;
    OSDEV_COMPONENTS_LOCKGUARD(generatorMutex);
    return uuidGenerator();
}

//static
MqttId MqttIdGenerator::nullId()
{
    return boost::uuids::nil_uuid();
}

// static
MqttId MqttIdGenerator::nameId(MqttId namespaceUuid, const std::string& name)
{
    boost::uuids::name_generator gen(namespaceUuid);
    return gen(name.c_str());
}

// static
MqttId MqttIdGenerator::nameId(const std::string& name)
{
    return MqttIdGenerator::nameId(mqttNamespace, name);
}
