/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_TOKEN_H
#define OSDEV_COMPONENTS_MQTT_TOKEN_H

// std
#include <ostream>
#include <string>
#include <cstdint>

// paho
#include <MQTTAsync.h>

namespace osdev {
namespace components {
namespace mqtt {

/*!
 *  \brief The Token class defines an operation token
 */
class Token
{
public:
    /*! @brief  Construct an invalid token.
     *  The token number is -1 in that case. The client is undefined, in this case empty.
     */
    Token();

    /*! @brief  Constructs token for an operation originating from specific client wrapper.
     *  @param  clientId    - Identifies the client wrapper
     *  @param  tokenNr     - Identifies the operation done on that client.
     */
    Token( const std::string &clientId, std::int32_t tokenNr );

    /*! @return True when token has a valid token number, false otherwise. */
    bool isValid() const { return -1 == m_token; }

    /*! @return The operation token */
    const std::string& clientId() const { return m_clientId; }

    /*! @return The operation token */
    std::int32_t token() const { return m_token; }

    /*! @return True if Tokens have the same clientId and token number, false otherwise. */
    bool equals( const Token &rhs ) const;

    /*!
     *  @brief Token is ordered.
     *  First on lexical test of clientId and with same clientId on token number.
     */
    bool smallerThan( const Token &rhs ) const;

private:
    std::string  m_clientId;             ///< Identified the client
    std::int32_t m_token;                ///< Identifies the operation on that client.
};

/**
 *  @return True if Tokens have the same clientId and token number, false otherwise.
 */
inline bool operator==( const Token &lhs, const Token &rhs )
{
    return lhs.equals( rhs );
}

inline bool operator==( const Token &lhs, std::int32_t rhs )
{
    return lhs.token() == rhs;
}

inline bool operator==( std::int32_t lhs, const Token &rhs )
{
    return lhs == rhs;
}

template <typename TLeft, typename TRight>
inline bool operator!=( const TLeft &lhs, const TRight &rhs )
{
    return !( lhs == rhs );
}

/*!
 *  @return True if Token lhs is smaller than token rhs
 */
inline bool operator<( const Token &lhs, std::int32_t rhs )
{
    return lhs.token() < rhs;
}

inline bool operator<( std::int32_t lhs, const Token &rhs )
{
    return lhs < rhs.token();
}

inline bool operator<( const Token &lhs, const Token &rhs )
{
    return lhs.smallerThan( rhs );
}

/*!
 *  @brief  Stream operator for a Token
 */
std::ostream& operator<<( std::ostream &os, const Token &rhs );

}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

#endif  // OSDEV_COMPONENTS_MQTT_TOKEN_H
