/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_SERVERSTATE_H
#define OSDEV_COMPONENTS_MQTT_SERVERSTATE_H

// std
#include <atomic>
#include <map>

// boost
#include <boost/signals2/connection.hpp>

// osdev::components::mqtt
#include "istatecallback.h"

namespace osdev {
namespace components {
namespace mqtt {

/*!
 *  \brief  Class for administrating ServerState callbacks.
 *          ServiceClientBase uses this object to notify the server state listeners
 *          of changes in the serverstate of the server that the client is connected to.
 */
class ServerState
{
public:
    /*!
     *  \brief Constructs a ServerState object. This object has a one to one relation with an IStateCallback object.
     *  \param stateCallbackIf identification of the interface that generates the signal.
     */
    explicit ServerState(const IStateCallback* stateCallbackIf);

    /*!
     *  \brief Destroy the ServerState.
     *         Calls clearAllStateChangeCallbacks() to unregister itself from the listeners.
     */
    virtual ~ServerState();

    // non copyable, non movable
    ServerState(const ServerState&)         = delete;
    ServerState& operator=(ServerState&)    = delete;
    ServerState(ServerState&&)              = delete;
    ServerState& operator=(ServerState&&)   = delete;

    /*!
     *  \brief  Registers a statechange callback method.
     *  \param  cb  - The callback method.
     *  \return handle that identifies the callback method.
     */
    StateChangeCallbackHandle registerStateChangeCallback(const IStateCallback::SlotStateChange& cb);

    /*!
     *  \brief  Unregisters a state change callback method.
     *  \param  handle Handle that identifies the callback method.
     */
    void unregisterStateChangeCallback(StateChangeCallbackHandle handle);

    /*!
     *  \brief  Removes all callback methods.
     *          An Unregister state is signalled to the listeners.
     */
    void clearAllStateChangeCallbacks();

    /*!
     *  \brief  Emit the State changed signal.
     *  \param  newState - The new state.
     */
    void emitStateChanged(StateEnum newState);

    /*!
     *  \brief  Return the last state received from server.
     *  \return state of server
     */
    StateEnum state() const;

    /*!
     *  \brief Returns the handle that will be given to the next callback that is registered.
     */
    static StateChangeCallbackHandle nextHandle()
    {
        return s_nextServerStateCallbackHandle;
    }

private:
    /*!
     *  Type for holding connections to server state callback functions.
     */
    using ServerStateCallbackMap = std::map<StateChangeCallbackHandle, boost::signals2::scoped_connection>;

    IStateCallback::SigStateChange sig_serverStateChanged;                          ///< Signal emitted when server state has changed.

    const IStateCallback* m_stateCallbackIf;                                        ///< Identification of the the interface that generates the signal (not owned)
    ServerStateCallbackMap m_serverStateCallbackMap;                                ///< Map with serverstate callback connections.

    static std::atomic<StateChangeCallbackHandle> s_nextServerStateCallbackHandle;  ///< Handle given to next serverstate callback registration (0 is not valid).

    StateEnum m_state;                                                              ///< Store the last state received from the server.

    /*!
     *  \return The clientId of the IStateCallback interface, or "null" if it is nullptr.
     */
    std::string stateCallbackClientId() const;

    static const std::string s_identifier;
};

}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

#endif  // OSDEV_COMPONENTS_MQTT_SERVERSTATE_H
