/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_SCOPEGUARD_H
#define OSDEV_COMPONENTS_MQTT_SCOPEGUARD_H

// std
#include <functional>

#include "macrodefs.h"
#include "utils.h"

#define OSDEV_COMPONENTS_SCOPEGUARD(variableName, ...)                                                            \
    osdev::components::mqtt::ScopeGuard OSDEV_COMPONENTS_COMBINE(Scope__Guard__##variableName##__, __LINE__)(__VA_ARGS__); \
    osdev::components::mqtt::apply_unused_parameters(OSDEV_COMPONENTS_COMBINE(Scope__Guard__##variableName##__, __LINE__));

namespace osdev {
namespace components {
namespace mqtt {

using CleanUpFunction = std::function<void()>;

/**
 * @brief Ensures that a cleanup function is called at the end of the current scope.
 */
class ScopeGuard
{
public:
    /**
     * @brief Constructs an empty scopeguard.
     * The scopeguard can be set by moving another ScopeGuard into this one.
     */
    ScopeGuard();

    /**
     * @brief Constructs a RAII instance that will call cleanupFunc in it's destructor.
     * @param cleanupFunc The cleanup function to call at the end of the current scope.
     * This cleanup function must not throw exceptions. If it does, the behavior is undefined.
     */
    ScopeGuard(const CleanUpFunction& cleanupFunc);

    // Movable, not copyable
    ScopeGuard(const ScopeGuard&) = delete;
    ScopeGuard& operator=(const ScopeGuard&) = delete;
    ScopeGuard(ScopeGuard&&) = default;
    ScopeGuard& operator=(ScopeGuard&&) = default;

    ~ScopeGuard() noexcept;

private:
    CleanUpFunction m_cleanupFunc;
};

}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

#endif  // OSDEV_COMPONENTS_MQTT_SCOPEGUARD_H
