/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_MQTTSUCCESS_H
#define OSDEV_COMPONENTS_MQTT_MQTTSUCCESS_H

// std
#include <string>
#include <vector>

// boost
#include <boost/variant.hpp>

// paho
#include <MQTTAsync.h>

// osdev::components::mqtt
#include "mqttmessage.h"

namespace osdev {
namespace components {
namespace mqtt {

/**
 * @brief Class that holds paho connection data which is returned in the connect response.
 */
class ConnectionData
{
public:
    /*!
     *  \brief  Construct an empty ConnectData instance.
     */
    ConnectionData();

    /*!
     *  \brief  Construct ConnectData based on incoming values from paho.
     *  \param  serverUri       - The serverUri to which the connection is made (needs to be copied).
     *  \param  mqttVersion     - The mqtt version used by the broker.
     *  \param  sessionPresent  - Flag that indicates if a session was present for the given clientId.
     */
    ConnectionData(char* serverUri, int mqttVersion, int sessionPresent);

    /*!
     *  \return The server uri.
     */
    const std::string& serverUri() const { return m_serverUri; }

    /*!
     *  \return The mqtt version.
     */
    int mqttVersion() const { return m_mqttVersion; }

    /*!
     *  \return if a session was present for the given clientId.
     */
    bool sessionPresent() const { return m_sessionPresent; }

private:
    std::string m_serverUri; ///< The broker server uri.
    int m_mqttVersion;       ///< The mqtt version used by the broker.
    bool m_sessionPresent;   ///< Flag that indicates whether a session was present for the client id used in the connect.
};

struct Unspecified
{
};

/*!
 *  \brief  Class for paho mqtt success response data.
 *          The paho success response data uses a union and can have different information depending on the command.
 */
class MqttSuccess
{
public:
    /*!
     *  \brief Response data for commands without specific data.
     *  \param token The token that identifies to which command this response belongs.
     */
    explicit MqttSuccess(MQTTAsync_token token);

    /*!
     *  \brief Response data for a subscribe command.
     *  \param token The token that identifies to which command this response belongs.
     *  \param qos Actual quality of service of the subscription.
     */
    MqttSuccess(MQTTAsync_token token, int qos);

    /*!
     *  \brief Response data for a subscribe many command.
     *  \param token The token that identifies to which command this response belongs.
     *  \param qosMany Actual quality of service of the subscription for each topic filter.
     */
    MqttSuccess(MQTTAsync_token token, const std::vector<int>& qosMany);

    /*!
     *  \brief Response data for a publish command.
     *  \param token The token that identifies to which command this response belongs.
     *  \param pubMsg The message that was published.
     */
    MqttSuccess(MQTTAsync_token token, const MqttMessage& pubMsg);

    /*!
     *  \brief Response data for a connect command.
     *  \param token The token that identifies to which command this response belongs.
     *  \param connData The connection data.
     */
    MqttSuccess(MQTTAsync_token token, const ConnectionData& connData);

    /*!
     *  \return the command token.
     */
    MQTTAsync_token token() const { return m_token; }

    /*!
     *  \return the qos
     *  \throw exception when command is not a subscribe command.
     */
    int qos() const;

    /*!
     *  \return a vector of qos values (matching the topics in the subscribe many command).
     *  \throw exception when command is not a subscribe many command.
     */
    std::vector<int> qosMany() const;

    /*!
     *  \return Message that has been published.
     *  \throw exception when command is not a publish command.
     */
    MqttMessage publishData() const;

    /*!
     *  return Connection data.
     *  throw exception when command is not a connect command.
     */
    ConnectionData connectionData() const;

private:
    MQTTAsync_token m_token;                                                                ///< Command token.
    boost::variant<int, std::vector<int>, MqttMessage, ConnectionData, Unspecified> m_data; ///< Data for the various commands.
};

}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

#endif  // OSDEV_COMPONENTS_MQTT_MQTTSUCCESS_H
