/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_MQTTSTREAM_H
#define OSDEV_COMPONENTS_MQTT_MQTTSTREAM_H

// This header is used in conjunction with mlogic/commmon/stringify.h to get output streaming of stl container types.
// The streaming operators are not suitable for marshalling because type information is lost!

// std
#include <array>
#include <list>
#include <map>
#include <ostream>
#include <set>
#include <string>
#include <vector>

namespace osdev {
namespace components {
namespace mqtt {

/**
 * @brief Streams a container with content for which stream operators are available.
 * @note This function is meant for printing and not for marshalling!
 * @tparam Open The container opening character.
 * @tparam Close The container closing character.
 * @tparam T The container type.
 * @param os The stream to use.
 * @param rhs The container that is to be streamed.
 * @param sep The field separator. Default is ", "
 * @return reference to the stream object.
 */
template <char Open, char Close, typename T>
std::ostream& streamContainer(std::ostream& os, const T& rhs, const std::string& sep = ", ")
{
    os << Open;
    for (auto cit = rhs.cbegin(); rhs.cend() != cit; ++cit) {
        os << *cit;
        if (std::next(cit) != rhs.end()) {
            os << sep;
        }
    }
    os << Close;

    return os;
}

}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

namespace std {

/**
 * @brief Streams a list that contains values for which an output stream operator is available.
 */
template <typename T>
std::ostream& operator<<(std::ostream& os, const std::list<T>& rhs)
{
    return osdev::components::mqtt::streamContainer<'<', '>'>(os, rhs);
}

/**
 * @brief Streams an array that contains values for which an output stream operator is available.
 */
template <typename T, std::size_t N>
std::ostream& operator<<(std::ostream& os, const std::array<T, N>& rhs)
{
    return osdev::components::mqtt::streamContainer<'[', ']'>(os, rhs);
}

/**
 * @brief Streams a vector that contains values for which an output stream operator is available.
 */
template <typename T>
std::ostream& operator<<(std::ostream& os, const std::vector<T>& rhs)
{
    return osdev::components::mqtt::streamContainer<'[', ']'>(os, rhs);
}

/**
 * @brief Streams a set that contains values for which an output stream operator is available.
 */
template <typename T>
std::ostream& operator<<(std::ostream& os, const std::set<T>& rhs)
{
    return osdev::components::mqtt::streamContainer<'{', '}'>(os, rhs);
}

/**
 * @brief Streams a map that contains keys and values for which an output stream operator is available.
 */
template <typename TKey, typename TValue>
std::ostream& operator<<(std::ostream& os, const std::map<TKey, TValue>& rhs)
{
    return osdev::components::mqtt::streamContainer<'{', '}'>(os, rhs);
}

/**
 * @brief Streams a pair that contains values for which an output stream operator is available.
 */
template <typename TFirst, typename TSecond>
std::ostream& operator<<(std::ostream& os, const std::pair<TFirst, TSecond>& rhs)
{
    os << "{" << rhs.first << " : " << rhs.second << "}";
    return os;
}

}       // End namespace std

#endif  // OSDEV_COMPONENTS_MQTT_MQTTSTREAM_H
