/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_MQTTMESSAGE_H
#define OSDEV_COMPONENTS_MQTT_MQTTMESSAGE_H

// std
#include <string>

// paho-c
#include <MQTTAsync.h>

namespace osdev {
namespace components {
namespace mqtt {

/*!
 *  @brief  Class for paho mqtt message data
 */
class MqttMessage
{
public:
    /*!
     * @brief   Construct empty MqttMessage instance
     */
    MqttMessage();

    /*!
     *  @brief  Construct MqttMessage instance by copying information gfrom the paho message struct
     *  @param  topic   - Paho topic data (copied)
     *  @param  msg     - Paho message data (copied)
     */
    MqttMessage( const std::string &topic, const MQTTAsync_message &msg );

    /*!
     *  @brief  Construct MqttMessage instance.
     *  @param  topic         - Topic String
     *  @param  retainedFlag  - Flag that indicates if message is retained
     *  @param  duplicateFlag - Flag that indicates if message is duplicate.
     *  @param  thePayload    - The message itself.
     */
    MqttMessage( const std::string &topic, bool retainedFlag, bool duplicateFlag, std::string thePayload );

    /*! @return The retained flag value. */
    bool retained() const { return m_retained; }

    /*! @return The duplicate flag value */
    bool duplicate() const { return m_duplicate; }

    /*! @return The topic on which the message is received. */
    const std::string& topic() const { return m_topic; }

    /*! @return The message payload. */
    const std::string& payload() const { return m_payload; }

    /*! @return This instance as a paho message */
    MQTTAsync_message toAsyncMessage() const;

private:
    bool            m_retained;     ///< Retained flag. Not all brokers communicate this flag correctly. (emqx does not, mosquitto does.)
    bool            m_duplicate;    ///< Duplicate flag ( for qos 1? )
    std::string     m_topic;        ///< The topic on which the message is recieved.
    std::string     m_payload;      ///< The actual message data.
};


}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

#endif  // OSDEV_COMPONENTS_MQTT_MQTTMESSAGE_H
