/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_IMQTTCLIENTIMPL_H
#define OSDEV_COMPONENTS_MQTT_IMQTTCLIENTIMPL_H

// std
#include <chrono>
#include <functional>
#include <ostream>
#include <set>
#include <string>
#include <vector>

// boost
#include <boost/optional.hpp>

// mlogic::mqtt
#include "connectionstatus.h"
#include "mqttmessage.h"
#include "mqtt_lwt.h"

namespace osdev {
namespace components {
namespace mqtt {

/**
 * @brief Interface that describes the minimal interface that a wrapper implementation needs to have in
 * order to coorporate with the MqttClient class.
 */
class IMqttClientImpl
{
public:
    virtual ~IMqttClientImpl();

    /**
     * @return id of this client.
     */
    virtual std::string clientId() const = 0;

    /**
     * @return The connection status of the wrapper.
     */
    virtual ConnectionStatus connectionStatus() const = 0;

    /**
     * @brief Connect the wrapper to the endpoint.
     * @param wait A flag that indicates if the method should wait for a succesful connection.
     * @return the operation token.
     */
    virtual std::int32_t connect( bool wait, const mqtt_LWT &lwt = mqtt_LWT() ) = 0;

    /**
     * @brief Disconnect the wrapper.
     * @param wait A flag that indicates if the method should wait for a succesful disconnect. When true the method will wait for the timeoutMs value plus some additional time.
     * @param timeoutMs A timeout in milliseconds. The timeout is used to give in flight messages a chance to get delivered. After the timeout the disconnect command is sent.
     * @return the operation token.
     */
    virtual std::int32_t disconnect(bool wait, int timeoutMs) = 0;

    /**
     * @brief Publish a message to an mqtt endpoint with a given quality of service.
     * When the connection is in state reconnect the publish is saved so that it can be published later.
     * @param message The message to send.
     * @param qos The quality of service to use (0, 1 or 2).
     * @return The message token. This token identifies the publication.
     */
    virtual std::int32_t publish(const MqttMessage& message, int qos) = 0;

    /**
     * @brief Publish messages when client is reconnected.
     */
    virtual void publishPending() = 0;

    /**
     * @brief Subscribe to a topic(filter).
     * The combination of topic and qos makes a subscription unique. Subscriptions with the same topic filter
     * but different qos are considered different, but they will overlap and cannot exist in the same wrapper.
     * @param topic The topic to subscribe to (can have wildcards). The topic cannot have overlap with existing topic subscriptions.
     * @param qos The quality of service to use (0, 1 or 2).
     * @param cb The callback that is called when messages are received for this subscription.
     * @return the operation token.
     */
    virtual std::int32_t subscribe(const std::string& topic, int qos, const std::function<void(MqttMessage msg)>& cb) = 0;

    /**
     * @brief Resubscribe existing topicfilters.
     * This method should be called only after a reconnect when subscriptions need to be reinstated.
     */
    virtual void resubscribe() = 0;

    /**
     * @brief Unsubscribe an existing subscription.
     * The combination of topic and qos make a subscription unique.
     * @param topic The topic to unsubscribe.
     * @param qos The quality of service of the subscription (0, 1 or 2).
     * @return the operation token.
     */
    virtual std::int32_t unsubscribe(const std::string& topic, int qos) = 0;

    /**
     * @brief Unsubscribe all subscriptions.
     */
    virtual void unsubscribeAll() = 0;

    /**
     * @brief Wait for commands to complete.
     * This method enables the user to wait for a set of tokens. An empty set means wait for all operations to complete.
     * @param waitFor The number of milliseconds to wait for completetion of all commands.
     * @param tokens The set of tokens to wait for.
     * @return The number of milliseconds left.
     */
    virtual std::chrono::milliseconds waitForCompletion(std::chrono::milliseconds waitFor, const std::set<std::int32_t>& tokens) const = 0;

    /**
     * @brief Check if a topic overlaps with existing topic subscriptions.
     * @param topic The topic to test.
     * @return true when overlap is detected, false otherwise.
     */
    virtual bool isOverlapping(const std::string& topic) const = 0;

    /**
     * @brief Check if a topic overlaps with existing topic subscriptions.
     * @param topic The topic to test.
     * @param[out] existingTopic Contains the topic on which overlap is detected.
     * @return true when overlap is detected. The existingTopic contains the topic on which overlap is detected, false otherwise.
     */
    virtual bool isOverlapping(const std::string& topic, std::string& existingTopic) const = 0;

    /**
     * @return A vector with the pending operation tokens.
     */
    virtual std::vector<std::int32_t> pendingOperations() const = 0;

    /**
     * @return true when client wrapper has pending subscriptions, false otherwise.
     */
    virtual bool hasPendingSubscriptions() const = 0;

    /**
     * @return The operation result.
     * @retval true operation has succeeded.
     * @retval false operation has failed.
     */
    virtual boost::optional<bool> operationResult(std::int32_t token) const = 0;
};

}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

#endif  // OSDEV_COMPONENTS_MQTT_IMQTTCLIENTIMPL_H
