/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_IMQTTCLIENT_H
#define OSDEV_COMPONENTS_MQTT_IMQTTCLIENT_H

// std
#include <chrono>
#include <functional>
#include <set>
#include <string>

// boost
#include <boost/optional.hpp>

// mlogic::client
#include "istatecallback.h"

// mlogic::mqtt
#include "connectionstatus.h"
#include "credentials.h"
#include "log.h"
#include "mqttmessage.h"
#include "mqtt_lwt.h"
#include "token.h"

namespace osdev {
namespace components {
namespace mqtt {

/**
 * @brief Interface that describes the minimal interface that a wrapper implementation needs to have in
 * order to coorporate with the MqttClient class.
 */
class IMqttClient : public virtual IStateCallback
{
public:
    virtual ~IMqttClient() {}

    /**
     * @brief Connect to the endpoint
     * @param host The host name or ip address.
     * @param port The port to use.
     * @param credentials The credentials to use.
     */
    virtual void connect( const std::string& host, int port, const Credentials& credentials,
                          const mqtt_LWT &lwt = mqtt_LWT(), bool blocking = false,
                          const log::LogSettings &log_settings = log::LogSettings() ) = 0;

    /**
     * @brief Connect to the endpoint
     * @param endpoint an uri endpoint description.
     */
    virtual void connect( const std::string& endpoint, const mqtt_LWT &lwt = mqtt_LWT(),
                          bool blocking = false, const log::LogSettings &log_settings = log::LogSettings() ) = 0;

    /**
     * @brief Disconnect the client from the broker
     */
    virtual void disconnect() = 0;

    /**
     * @brief Publish a message with a given quality of service (0, 1 or 2).
     * @param message The message to publish.
     * @param qos The quality of service to use.
     * @return The token that identifies the publication (used in the deliveryCompleteCallback).
     */
    virtual Token publish(const MqttMessage& message, int qos) = 0;

    /**
     * @brief Subscribe to a topic(filter).
     * The combination of topic and qos makes a subscription unique on a wrapper client. When a topic has overlap
     * with an existing subscription it is guaranteed that the given callback is only called once.
     * @param topic The topic to subscribe to (can have wildcards). The topic can have overlap with existing topic subscriptions.
     * @param qos The quality of service to use (0, 1 or 2).
     * @param cb The callback that is called when messages are received for this subscription.
     * @return A token that identifies the subscribe command.
     */
    virtual Token subscribe(const std::string& topic, int qos, const std::function<void(MqttMessage)>& cb) = 0;

    /**
     * @brief Unsubscribe an existing subscription.
     * The combination of topic and qos can occur on multiple wrapper clients. All subscriptions that match are unsubscribed.
     * @param topic The topic to unsubscribe.
     * @param qos The quality of service of the subscription (0, 1 or 2).
     * @return A set of unsubscribe tokens identifying the unsubscribe commands. Usually the set will contain only one item.
     */
    virtual std::set<Token> unsubscribe(const std::string& topic, int qos) = 0;

    /**
     * @brief Wait for all commands to complete.
     * @param waitFor The number of milliseconds to wait for completetion of all commands.
     * @return True when all commands have completed in time or false if not.
     */
    virtual bool waitForCompletion(std::chrono::milliseconds waitFor) const = 0;

    /**
     * @brief Wait for a single command to complete.
     * @param waitFor The number of milliseconds to wait for completetion of the command.
     * @param token The token to wait for.
     * @return True when the command has completed in time or false if not.
     * @note Non existent tokens are also reported as completed.
     */
    virtual bool waitForCompletion(std::chrono::milliseconds waitFor, const Token& token) const = 0;

    /**
     * @brief Wait for commands to complete.
     * This method enables the user to wait for a set of commands. An empty set means to wait for all commands to complete.
     * @param waitFor The number of milliseconds to wait for completetion of all commands.
     * @param tokens The tokens to wait for. An empty set means to wait for all commands to complete.
     * @return True when the commands have completed in time or false if not.
     * @note Non existent tokens are also reported as completed.
     */
    virtual bool waitForCompletion(std::chrono::milliseconds waitFor, const std::set<Token>& tokens) const = 0;

    /**
     * @brief Get the result of a command.
     * @param token The token identifying the result.
     * @return The command result when available.
     */
    virtual boost::optional<bool> commandResult(const Token& token) const = 0;

    /**
     * @return The endpoint uri.
     */
    virtual std::string endpoint() const = 0;
};

}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

#endif  // OSDEV_COMPONENTS_MQTT_IMQTTCLIENT_H
