/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_MEASUREMENT_IHISTOGRAM_H
#define OSDEV_COMPONENTS_MQTT_MEASUREMENT_IHISTOGRAM_H

// std
#include <ostream>
#include <string>
#include <vector>

namespace osdev {
namespace components {
namespace mqtt {
namespace measurement {

/**
 * @brief Class that holds the dynamic part of a histogram.
 * Used to take a snapshot of the histogram.
 */
class HistogramData
{
public:
    /**
     * @brief Construct dynamic histrogram data.
     * @param smallestValue A stringified version of the smallest recorded value since the last update.
     * @param largestValue A stringified version of the largest recorded value since the last update.
     * @param smallestValueSinceLastClear A stringified version of the smallest value as string since last clear.
     * @param largestValueSinceLastClear  A stringified version of the largest value as string since last clear.
     * @param averageValueSinceLastClear A stringified version of the average value as string since last clear.
     * @param nrOfValuesSinceLastClear Number of values since last clear.
     * @param histCounts The histogram counts.
     */
    HistogramData(const std::string& smallestValue, const std::string& largestValue,
        const std::string& smallestValueSinceLastClear,
        const std::string& largestValueSinceLastClear,
        const std::string& averageValueSinceLastClear,
        std::size_t nrOfValuesSinceLastClear,
        const std::vector<std::size_t>& histCounts)
        : m_smallestValueString(smallestValue)
        , m_largestValueString(largestValue)
        , m_smallestValueSinceLastClearString(smallestValueSinceLastClear)
        , m_largestValueSinceLastClearString(largestValueSinceLastClear)
        , m_averageValueSinceLastClearString(averageValueSinceLastClear)
        , m_numberOfValuesSinceLastClear(nrOfValuesSinceLastClear)
        , m_data(histCounts)
    {
    }

    /**
     * @return The smallest recorded value as a string.
     */
    const std::string& smallestValueString() const
    {
        return m_smallestValueString;
    }

    /**
     * @return The largest recorded value as a string.
     */
    const std::string& largestValueString() const
    {
        return m_largestValueString;
    }

    /**
     * @return The number of value since last clear.
     */
    std::size_t numberOfValuesSinceLastClear() const
    {
        return m_numberOfValuesSinceLastClear;
    }

    /**
     * @return The smallest value since last clear as a string.
     */
    const std::string& smallestValueSinceLastClearString() const
    {
        return m_smallestValueSinceLastClearString;
    }

    /**
     * @return The largest value since last clear as a string.
     */
    const std::string& largestValueSinceLastClearString() const
    {
        return m_largestValueSinceLastClearString;
    }

    /**
     * @return The average value since last clear as a string.
     */
    const std::string& averageValueSinceLastClearString() const
    {
        return m_averageValueSinceLastClearString;
    }

    /**
     * @return The histogram counts.
     */
    const std::vector<std::size_t>& data() const
    {
        return m_data;
    }

private:
    std::string m_smallestValueString;
    std::string m_largestValueString;
    std::string m_smallestValueSinceLastClearString;
    std::string m_largestValueSinceLastClearString;
    std::string m_averageValueSinceLastClearString;
    std::size_t m_numberOfValuesSinceLastClear;
    std::vector<std::size_t> m_data;
};

/**
 * @brief Interface for histogram classes
 * Can be used to store histograms in a container and visualize the histogram.
 */
class IHistogram
{
public:
    virtual ~IHistogram();

    /**
     * @return The histogram identification (title).
     */
    virtual const std::string& id() const = 0;

    /**
     * @return The number of buckets.
     */
    virtual std::size_t numBuckets() const = 0;

    /**
     * @return The bucket width as a floating point value.
     */
    virtual double bucketWidth() const = 0;

    /**
     * @return The minimum value of the histogram definition as a string.
     */
    virtual std::string minValueString() const = 0;

    /**
     * @return The maximum value of the histogram definition as a string.
     */
    virtual std::string maxValueString() const = 0;

    /**
     * @return The value unit as a a string.
     */
    virtual const std::string& unit() const = 0;

    /**
     * @return The histogram data.
     * This takes a snapshot of the histogram data.
     */
    virtual HistogramData histogramData() const = 0;

    /**
     * @return The number of value since last clear.
     */
    virtual std::size_t numberOfValuesSinceLastClear() const = 0;

    /**
     * @return The smallest value since last clear as a string.
     */
    virtual std::string smallestValueSinceLastClear() const = 0;

    /**
     * @return The largest value since last clear as a string.
     */
    virtual std::string largestValueSinceLastClear() const = 0;

    /**
     * @return The average value since last clear as a string.
     */
    virtual std::string averageValueSinceLastClear() const = 0;

    /**
     * @brief Clears the values that are kept between successive clearRunningValues calls.
     * These are:
     *  smallestValueSinceLastClear
     *  largestValueSinceLastClear
     *  averageValueSinceLastClear
     */
    virtual void clearRunningValues() = 0;

    /**
     * @return The ascii representation of the histogram.
     */
    virtual std::string toString() const = 0;
};

/**
 * @brief Make an ascii visualisation of the given histogram data.
 * @param histogram The histogram to visualize.
 */
std::string visualizeHistogram(const IHistogram& histogram);

/**
 * @brief Stream operator for IHistogram instances.
 */
std::ostream& operator<<(std::ostream& os, const IHistogram& rhs);

}       // End namespace measurement
}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

#endif  // OSDEV_COMPONENTS_MQTT_MEASUREMENT_IHISTOGRAM_H
