/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include <chrono>
#include <type_traits>

// function std::chrono::ceil is added in the c++17 standard.
// Added implementation from https://en.cppreference.com/w/cpp/chrono/duration/ceil
// to this compatibility header. The header only defines this implementation when
// the c++ compiler is used with an older standard.

#if !defined __cpp_lib_chrono || __cpp_lib_chrono < 201611

namespace std {
namespace chrono {

template <class T>
struct is_duration : std::false_type
{
};
template <class Rep, class Period>
struct is_duration<
    std::chrono::duration<Rep, Period>> : std::true_type
{
};

template <class To, class Rep, class Period,
    class = typename enable_if<is_duration<To>{}>::type>
To ceil(const std::chrono::duration<Rep, Period>& d)
{
    To t = std::chrono::duration_cast<To>(d);
    if (t < d)
    {
        return t + To{ 1 };
    }

    // or else...
    return t;
}

}       // End namespace chrono
}       // End namespace std

#endif
