/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_MQTT_COMMONDEFS_H
#define OSDEV_COMPONENTS_MQTT_COMMONDEFS_H

// std
#include <chrono>
#include <cstdint>
#include <map>
#include <ostream>
#include <set>
#include <string>
#include <vector>

// boost
#include <boost/lexical_cast.hpp>
#include <boost/optional.hpp>
#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_io.hpp>

#include "utils.h"


/// @brief Check if an id is valid
/// @throws InvalidArgumentException if id is invalid
#define OSDEV_COMPONENTS_CHECKMQTTID(id)                                                                \
    [&] {                                                                                              \
        if (id.is_nil()) {                                                                             \
        }                                                                                              \
    }()


namespace osdev {
namespace components {
namespace mqtt {

using MqttId = boost::uuids::uuid;
using OptionalId = MqttId;
using MqttIdSet = std::set<MqttId>;
using MqttIdSetIterator = MqttIdSet::const_iterator;
using MqttIdSetDelta = std::pair<MqttIdSet, MqttIdSet>;
using StdTime = std::chrono::system_clock::time_point;
using OptionalTime = boost::optional<StdTime>;
using StdTimeVec = std::vector<StdTime>;
using SequenceNumber = std::uint32_t;
using OptionalSequenceNumber = boost::optional<SequenceNumber>;
using CustomField = std::string;
using CustomFieldCollection = std::vector<CustomField>;

using CountryCodeEnum = std::int32_t;

/**
 * @brief Defines a parsed uri.
 */
using ParsedUri = std::map<std::string, std::string>;

/**
 * @brief Type for the parsed query part of a uri.
 */
using ParsedQuery = std::map<std::string, std::string>;

/**
 * @brief Type for the parsed path part of a uri.
 */
using ParsedPath = std::vector<std::string>;

/**
 * @brief Defines a duration with the granularity of a day in seconds (24 * 60 * 60 = 86400).
 * This duration can be used to create a time_point at midnight of a given DateTime amongst others.
 *
 * The representation is a signed type so that negative durations are also possible.
 */
using days = std::chrono::duration<std::int32_t, std::ratio<86400>>;

/**
 * @brief Defines a duration with the granularity of a year in seconds. A year is a bit longer than 365 days (365.2425). If a year is
 * subtracted from a date the time part of the new date will therefore differ from the time part of the subtracted from date.
 *
 * The representation is a signed type so that negative durations are also possible.
 */
using years = std::chrono::duration<std::int32_t, std::ratio<31556952>>; // excactly 365 days would give 31536000

/**
 * A timepoint type that is printed with millisecond resolution.
 */
struct StdTimeMs
{
    StdTimeMs(const StdTime& tp)
        : timePoint(tp)
    {
    }

    operator StdTime() const { return timePoint; }

    StdTime timePoint;
};

std::ostream& operator<<(std::ostream& os, const StdTimeMs& rhs);

}       // End namespace mqtt
}       // End namespace components
}       // End namespace osdev

namespace std {

std::ostream& operator<<(std::ostream& os, const osdev::components::mqtt::StdTime& rhs);

}       // End namespace std

#endif  // OSDEV_COMPONENTS_MQTT_COMMONDEFS_H
