/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

// std
#include <iostream>
#include <unistd.h>
#include <chrono>

#include "subscriber.h"

const int MAX_LOOP_COUNT = 10;

enum TIME_RES
{
    T_MICRO,
    T_MILLI,
    T_SECONDS
};

std::uint64_t getEpochUSecs()
{
    auto tsUSec =std::chrono::time_point_cast<std::chrono::microseconds>(std::chrono::system_clock::now());
    return static_cast<std::uint64_t>(tsUSec.time_since_epoch().count());
}


void sleepcp( int number, TIME_RES resolution = T_MILLI )    // Cross-platform sleep function
{
    int factor = 0; // Should not happen..

    switch( resolution )
    {
        case T_MICRO:
            factor = 1;
            break;

        case T_MILLI:
            factor = 1000;
            break;

        case T_SECONDS:
            factor = 1000000;
        break;
    }

    usleep( number * factor );
}

int main( int argc, char* argv[] )
{
    int loop_counter = 0;
    // Satisfy the compiler
    (void)argc;
    (void)argv;

    std::cout << "Creating the subscriber : ";
    // Create the subscriber
    Subscriber *pSubscriber = new Subscriber( "Test_Subscriber" );
    if( pSubscriber != nullptr )
    {
        std::cout << "[OK]" << std::endl;
        std::cout << "Connecting to the test-broker : " << std::endl;
        pSubscriber->connect( "localhost", 1883, "", "", "test/subscriber/LWT", "Subscriber disconnected." );

        std::cout << "Subscribing to the test-topic....." << std::endl;
        pSubscriber->subscribe( "test/publisher/TestPublisher_0" );
        pSubscriber->subscribe( "test/publisher/TestPublisher_1" );
        pSubscriber->subscribe( "test/publisher/TestPublisher_2" );
        pSubscriber->subscribe( "test/publisher/TestPublisher_3" );

        // Start a loop to give the subscriber the possibility to do its work.
        while (loop_counter < MAX_LOOP_COUNT)
        {
            sleepcp( 1, T_SECONDS );   // Sleep 1 Sec to give the scheduler the change to interfene.
            loop_counter++;
        }

        std::cout << "Unsubscribing from test/publisher/#" << std::endl;

        pSubscriber->unsubscribe("test/publisher/#" );
        pSubscriber->unsubscribe( "test/publisher/TestPublisher_0" );
        pSubscriber->unsubscribe( "test/publisher/TestPublisher_1" );
        pSubscriber->unsubscribe( "test/publisher/TestPublisher_2" );
        pSubscriber->unsubscribe( "test/publisher/TestPublisher_3" );

        // Start a loop to give the subscriber the possibility to do its work.
        while (loop_counter < MAX_LOOP_COUNT)
        {
            sleepcp( 1, T_SECONDS );   // Sleep 1 Sec to give the scheduler the change to interfene.
            loop_counter++;
        }

        pSubscriber->subscribe("test/publisher/#");

        while (1)
        {
            sleepcp(1, T_MILLI);   // Sleep 1 mSec to give the scheduler the change to interfene.
        }

    }
    else
    {
        std::cout << "[FAILED]" << std::endl;
        return -1;
    }
    return 0;
}
