/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

// std
#include <iostream>
#include <unistd.h>
#include <chrono>
#include <vector>

#include "publisher.h"
#include "subscriber.h"

enum TIME_RES
{
    T_MICRO,
    T_MILLI,
    T_SECONDS
};

std::vector<std::string> positionTable = {
    "1", "99", "2", "98", "3", "97", "4", "96", "5", "95",
    "6", "94", "7", "93", "8", "92", "9", "91",
    "10", "90", "11", "89", "12", "88", "13", "87", "14", "86",
    "15", "85", "16", "84", "17", "83", "18", "82", "19", "81",
    "20", "80", "21", "79", "22", "78", "23", "77", "24", "76",
    "25", "75", "26", "74", "27", "73", "28", "72", "29", "71",
    "30", "70", "31", "69", "32", "68", "33", "67", "34", "66",
    "35", "65", "36", "64", "37", "63", "38", "62", "39", "61",
    "40", "60", "41", "59", "42", "58", "43", "57", "44", "56",
    "45", "55", "46", "54", "47", "53", "48", "52", "49", "51",
    "50", "50", "51", "49", "52", "48", "53", "47", "54", "46",
    "55", "45", "56", "44", "57", "43", "58", "42", "59", "41",
    "60", "40", "61", "39", "62", "38", "63", "37", "64", "36",
    "65", "35", "66", "34", "67", "33", "68", "32", "69", "31",
    "70", "30", "71", "29", "72", "28", "73", "27", "74", "26",
    "75", "25", "76", "24", "77", "23", "78", "22", "79", "21",
    "80", "20", "81", "19", "82", "18", "83", "17", "84", "16",
    "85", "15", "86", "14", "87", "13", "88", "12", "89", "11",
    "90", "10", "91", "9", "92", "8", "93", "7", "94", "6",
    "95", "5", "96", "4", "97", "3", "98", "2", "99", "1",
    "10", "90", "20", "80", "30", "70", "40", "60", "50", "50",
    "60", "40", "70", "30", "80", "20", "90", "10"
};

std::uint64_t getEpochUSecs()
{
    auto tsUSec =std::chrono::time_point_cast<std::chrono::microseconds>(std::chrono::system_clock::now());
    return static_cast<std::uint64_t>(tsUSec.time_since_epoch().count());
}


void sleepcp( int number, TIME_RES resolution = T_MILLI )    // Cross-platform sleep function
{
    int factor = 0; // Should not happen..

    switch( resolution )
    {
        case T_MICRO:
            factor = 1;
            break;

        case T_MILLI:
            factor = 1000;
            break;

        case T_SECONDS:
            factor = 1000000;
        break;
    }

    usleep( number * factor );
}

int main( int argc, char* argv[] )
{
    // We're not using the command parameters, so we just want to keep the compiler happy.
    (void)argc;
    (void)argv;


    // Create the publisher, run it and publish a control message every 35 sec.
    int publishInterval = 30;
    std::string ld_topic = "priva/0/devices/actuators/geiger-lamel-1";

    std::cout << "Create the subscriber object : ";
    Subscriber *pSubscriber = new Subscriber( "TestSubscriber" );
    if( pSubscriber != nullptr )
    {
        std::cout << "{OK}" << std::endl;
        std::cout << "Connecting to the broker : ";
        pSubscriber->connect( "192.168.1.110", 1883, "", "", "LWT-subscriber", "connection disrupted.." );
        pSubscriber->subscribe( ld_topic );
    }

    std::cout << "Create the publisher object : ";
    Publisher *pPublisher = new Publisher();
    if( pPublisher != nullptr )
    {
        std::cout << "{OK}" << std::endl;
        std::cout << "Connecting to the broker : ";
        pPublisher->connect( "192.168.1.110", 1883, "", "", "LWT-publisher", "connection disrupted.." );

        // Assume we are connected now, start publishing.
        while( 1 )
        {
            // Follow different patterns with bigger jumps.
            for( unsigned int nCount = 0; nCount < positionTable.size(); nCount++ )
            {
                std::cout << "==============================================================================" << std::endl;
                std::string payload = "{\"version\": 1, \"id\": \"geiger-lamel-1\", \"requesterId\": \"me\", \"instruction\": \"SETPOINT\", \"desiredValue\": " + positionTable[nCount] + ", \"requestedAt\": " + std::to_string( getEpochUSecs() ) + "}";
                pPublisher->publish( std::string( "priva/1100000916/devices/actuators/geiger-lamel-1/control"), payload );
                sleepcp(publishInterval, T_SECONDS);
            }
        }
    }
    else
        return -1;

}
