/****************************************************************************
 * Copyright (c) 2022 Priva b.v.
 ****************************************************************************/

#pragma once

// std
#include <string>
#include <variant>
#include <vector>

using modbusData = std::vector<std::variant<uint8_t, uint16_t>>;

// Forward Declaration
class ConnectionConfig;

/*!
 * The IModbusAdapter class provides an abstract way of using
 *          the modbus stack. Implemented as a pure virtual, it cannot be instantiated.
 *          This represents a unique connection to either a serial bus or a TCP connection.
 */
class IModbusAdapter
{

public:
    /// Default DTor.
    virtual ~IModbusAdapter() = default;

    /// Create a modbus connection, accepting a configuration object.
    virtual bool ModbusConnect( const ConnectionConfig &conncfg ) = 0;

    /*!
     * ModbusDisconnect
     *          Disconnect from the serial bus or the TCP connection, freeing its resources
     */
    virtual bool ModbusDisconnect() = 0;

    /*!
     * Read data from a modbus device given by its parameters.
     * @param   slaveId         - The Id of the ModbusDevice.
     * @param   functionCode    - The code describing the action we want to perform on the device.
     *                            Given by an enum, provided by the modbus-stack.
     * @param   startAddress    - Startaddres of the register we want to read.
     * @param   noOfItems       - The number of items we expect back.
     * @returns modbusData      - A vector holding each register in an entry in the same order as they are received.
     *                            Empty if no data was received.
     */
    virtual modbusData ModbusReadData( int slaveId, int functionCode, int startAddress, int noOfItems ) = 0;

    /*!
     * Read data from the holdregisters ( or keep-registers ) of a modbus device.
     * @param   slaveId         - The Id of the ModbusDevice.
     * @param   startAddress    - Startaddres of the register we want to read.
     * @param   noOfItems       - The number of items we expect back.
     * @returns modbusData      - A vector holding each register in an entry in the same order as they are received.
     *                            Empty if no data was received.
     */
    virtual modbusData ModbusReadHoldReg( int slaveId, int startAddress, int noOfItems ) = 0;

    /*!
     * Write data to the device.
     * @param slaveId           - The Id of the Modbus device
     * @param funtionCode       - The code describing the action we want to perform on the device
     *                            given by an enum, provided by the modbus-stack.
     * @param startAddress      - Startaddres of the register we want to read.
     * @param noOfItems         - The number of items we expect to be written
     * @param values            - The values we want to write to the given device. Each vector-entry represents a single byte
     *                            and they will be sent in sequence.
     */
    virtual void ModBusWriteData( int slaveId, int functionCode, int startAddress, int noOfItems, std::vector<int>values ) = 0;

    /*!
     * Indicates if this connection is alive.
     * @return  True if alive, false if not.
     */
    virtual bool isConnected() const = 0;

    /*!
     * returns the translated error code coming from the modbus stack.
     * @param   errnum          - error Number coming from the stack.
     * @return  The translated error as a string. Empty if not resolvable.
     */
    virtual std::string ErrorString( int errnum ) const = 0;
};
