/****************************************************************************
 * Copyright (c) 2022 Priva b.v.
 ****************************************************************************/
#pragma once

#include <string>
#include <unordered_map>

/*!
 * The ConnectionPort enum.
 *          CP_EXTERNAL - the first serial port
 *          CP_IOBUS    - The second serial port through RJ45 connector
 *          CP_TCP      - TCP Connections
 */
enum class ConnectionPort : unsigned int
{
    CP_EXTERNAL     = 0,
    CP_IOBUS,
    CP_TCP
};

/*!
 * The Parity enum.
 *          Used in a serial port context
 */
enum class Parity : unsigned int
{
    PAR_ODD,
    PAR_EVEN,
    PAR_NONE
};

/*!
 * The ConnectionType enum.
 *          Added for convenience, to distinguish between types.
 */
enum class ConnectionType : unsigned int
{
    CT_SERIAL,
    CT_TCP,
    CT_UNKNOWN
};

/*!
 * The ConnectionConfig class holds all the information we need to establish a proper connection.
 *        It can be created by a configuration object and passed on to the ModBus stack.
 *        By using this class, all the connectioninfo is within its context for convenience.
 *
 *        Data can be accessed by their resp. getter-methods to avoid internal data-exposure
 *        according to the basic principles of object oriented programming. ( Abstraction, Encapsulation, Inheritance, and Polymorphism )
 */
class ConnectionConfig
{
public:
    /*!
     * ConnectionConfig Constructor. Used to create a new serial connection.
     * @param port      - The portname given by its enum.
     * @param baud      - The port speed in a serial port context ( default = 115200 )
     * @param parity    - The parity. ( Default : None, no parity )
     * @param dataBits  - The number of databits. RTU uses 8 (0 - 255), ASCII uses 7 (0 - 127). Default is RTU
     * @param stopBits  - The number of stopbits used to detect the end of the frame. ( Default = 1 )
     * @param timeOut   - Timeout in .1 secs. See the termios documentation for deviations on this.
     */
    ConnectionConfig( ConnectionPort port, int baud = 115200, Parity parity = Parity::PAR_NONE, int dataBits = 8, int stopBits = 1, int timeOut = -1 )
        : port( port )
        , baudRate( baud )
        , parity( parity )
        , dataBits( dataBits )
        , stopBits( stopBits )
        , ipaddress()
        , portnumber( -1 )
        , timeOut( timeOut )
        , connectionType( ConnectionType::CT_SERIAL )
    {}

    /*!
     * \brief ConnectionConfig Constructor. Used to create a new TCP connection.
     * @param port      - The portname given by its enaum. ( Should be CP_TCP )
     * @param ip        - The ip address of the ModBus device we want to connect to.
     * @param portnum   - The portnumber the ModBus device is using
     * @param timeOut   - Timeout in which a modbus device should respond.
     */
    ConnectionConfig( ConnectionPort port, const std::string &ip, int portnum, int timeOut = -1  )
        : port( port )
        , baudRate( -1 )
        , parity( Parity::PAR_NONE )
        , dataBits( -1 )
        , stopBits( -1 )
        , ipaddress( ip )
        , portnumber( portnum )
        , timeOut( timeOut )
        , connectionType( ConnectionType::CT_TCP )
    {}

    // Getters and Setters. Implemented to avoid outside meddling on the member variables.
    std::string getPort() const { return portMap.at(port); }      ///< Get the translated portName.
    ConnectionPort getPortEnum() const { return port; }           ///< Get the portname Enum
    int getBaudRate() const { return baudRate; }                  ///< Get the given baudrate as int.
    char getParity() const { return parityMap.at( parity ); }     ///< Get the translated parity.
    Parity getParityEnum() const { return parity; }               ///< Get the parity Enum
    int getDataBits() const { return dataBits; }                  ///< Get the number of databits ( 7 for ASCII, 8 for RTU )
    int getStopBits() const { return stopBits; }                  ///< Get the number of stopBits. ( de-facto = 1 )

    std::string getIpAddress() const { return ipaddress; }        ///< Get the ip-address as string
    int getTcpPort() const { return portnumber; }                 ///< Get the tcp portnumber as int

    int getTimeOut() const { return timeOut; }                    ///< Get the timeout as a multitude of 0.1 sec.
    ConnectionType getType() const { return connectionType; }     ///< Get the connection type ( Serial, TCP or Unknown )

private:

    /// Serial connections
    ConnectionPort  port;                                         ///< Member variable holding the portName Enum
    int             baudRate;                                     ///< Member variable holding the Serial port Baudrate
    Parity          parity;                                       ///< Member variable holding the Serial port Parity
    int             dataBits;                                     ///< Member variable holding the number of databits
    int             stopBits;                                     ///< Member variable holding the number of stopbits

    /// TCP connections
    std::string     ipaddress;                                    ///< Member variable holding the ip-address of the TCP-connection
    int             portnumber;                                   ///< Member variable holding the portnumber of the TCP-connection

    /// Generic
    int             timeOut;                                      ///< Member variable holding the timeOut value
    ConnectionType  connectionType;                               ///< Member variable holding the connection type.

    /// Translation tables for portnames and parity.
    // ============================================================
    // == Change accordingly to the devicenames on your platform ==
    // ============================================================
    std::unordered_map<ConnectionPort, std::string> portMap =
    {
        { ConnectionPort::CP_EXTERNAL, "/dev/ttyUSB0" },
        { ConnectionPort::CP_IOBUS, "/dev/ttyUSB1" }
    };

    std::unordered_map<Parity, char> parityMap =
    {
        { Parity::PAR_EVEN, 'E' },
        { Parity::PAR_ODD, 'O' },
        { Parity::PAR_NONE, 'N' }
    };
};
