/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_THREADCONTEXT_H
#define OSDEV_COMPONENTS_THREADCONTEXT_H

// Qt
#include <QString>

namespace osdev {
namespace components {

/**
 * @brief Set the current thread context.
 * The context is restored to the previous context when this object goes out of scope.
 * @note This object is meant to be used on the stack.
 */
class ThreadContextScope
{
public:
    /**
     * @brief Construct a scoped object that sets the current thread context.
     * @param context The context that will be used by the logging framework.
     */
    explicit ThreadContextScope(const QString& context);
    ~ThreadContextScope();

    // Non copyable and non movable
    ThreadContextScope(const ThreadContextScope&) = delete;
    ThreadContextScope& operator=(const ThreadContextScope&) = delete;
    ThreadContextScope(ThreadContextScope&&) = delete;
    ThreadContextScope& operator=(ThreadContextScope&&) = delete;

private:
    QString m_previousContext;  ///< Copy of the previous context.
};

/**
 * @brief Add context to a thread.
 * For every thread only one specific instance of this object will exist.
 * @note Contexts can only be set by using a ThreadContextScope object.
 */
class ThreadContext
{

// Contexts can only be set by using a ThreadContextScope object.
friend class ThreadContextScope;

public:
    static ThreadContext& instance();

    /**
     * @brief Return the thread context.
     */
    const QString& context() const
    {
        return m_context;
    }

private:
    /**
     * @brief Set the thread context.
     */
    void setContext(const QString& contextString)
    {
        m_context = contextString;
    }

    /**
     * Construct a ThreadContext object.
     * The context is set to "default"
     */
    ThreadContext();

    // Non copyable and non movable
    ThreadContext(const ThreadContext&) = delete;
    ThreadContext& operator=(const ThreadContext&) = delete;
    ThreadContext(ThreadContext&&) = delete;
    ThreadContext& operator=(ThreadContext&&) = delete;

    QString m_context;  ///< The context string
};

}   /* End namespace components */
}   /* End namespace osdev */

#endif // OSDEV_COMPONENTS_THREADCONTEXT_H
