/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

#ifndef OSDEV_COMPONENTS_JOBSCHEDULER_H
#define OSDEV_COMPONENTS_JOBSCHEDULER_H

#include "jobdata.h"
#include "jobtimer.h"

#include <QObject>
#include <QUuid>
#include <QHash>
#include <QString>
#include <QPointer>

namespace osdev {
namespace components {
/*
 *  ________________________________________
 * / Real computer scientists despise the   \
 * | idea of actual hardware. Hardware has  |
 * | limitations, software doesn't. It's a  |
 * | real shame that Turing machines are so |
 * \ poor at I/O.                           /
 * ----------------------------------------
 *   \
 *    \
 *        .--.
 *       |o_o |
 *       |:_/ |
 *      //   \ \
 *     (|     | )
 *    /'\_   _/`\
 *    \___)=(___/
 *//*!
 * \brief   Schedule jobs to be executed on a specific Date / Time or on a regular interval.
 *          At this moment, it will schedule in intervals of 24 hours, if the date is set in the past.
 *          If the dat is in the future, it will be run for the first time on that specific date / Time
 *          and after that every 24 hours.
 */
class JobScheduler : public QObject
{
    Q_OBJECT

public:
    /*!
     * \brief JobScheduler  Constructor.
     * \param _parent   - The object creating the JobScheduler.
     *                    By adding this pointer, we use the cascade deletion construction that comes with Qt.
     */
    JobScheduler( QObject *_parent = nullptr );

    /*!
     * \brief JobScheduler  Constructor
     * \param lstJobs     - The list of jobs, coming from the configuration that we want to have scheduled.
     * \param _parent     - The object creating the JobScheduler.
     *                      By adding this pointer, we use the cascade deletion construction that comes with Qt.
     */
    JobScheduler( const QList < osdev::components::JobData > &lstJobs, QObject * _parent = nullptr );

    /*!
     * \brief   Schedule the job based on the parameters and settings.
     * \param   _job_name       - The name of this job used for registration
     * \param   _param_list     - Parameters by the job.
     * \param   _interval       - The interval this job should be scheduled to.
     * \param   _target_object  - The plugin or object this job is intended for. Used as input to the pluginmanager.
     * \param   _run_time       - The time this job should run. Empty or '*' if interval based.
     * \param   _run_date       - The date this job should run. Empty or '*' if daily.
     */
    void scheduleJob( const QString& _job_name,
                      const QHash<QString, QVariant>& _param_list,
                      const int _interval,
                      const QString& _target_object,
                      const QTime& _run_time,
                      const QDate& _run_date = QDate::currentDate() );

    /*!
     * \brief   Added for convenience. It will search for the given name and try to reschedule the job.
     */
    void rescheduleJob( const QString& _jobName );

    /*!
     * \brief   Start and initialize the jobscheduler.
     *          All configured Jobs are started and reported to the logfile.
     */
    void start();

private:
    /*!
     *  \brief  Create the requested job and add it to the Schedule list.
     *  \param  _jobItem    - Structure containing the data needed for this job.
     */
    void createJob( const JobData& _jobItem );

    QHash<QString, QPointer<JobTimer>>  m_jobSchedule;  ///<    The scheduler list

private slots:
    /*!
     * \brief   Called by the TimerObjects. It's identification (jobName) is used to determine
     *          which job has to be run
     */
    void slotRunJob( const QString& jobName );

signals:
    void signalRunJob( const JobData& data );

};

}   /* End namespace components */
}   /* End namespace osdev */

#endif  /* OSDEV_COMPONENTS_JOBSCHEDULER_H */
