/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

#include "jobscheduler.h"
#include "log.h"

using namespace osdev::components;

JobScheduler::JobScheduler( QObject *_parent )
    : QObject( _parent )
    , m_jobSchedule()
{

}

JobScheduler::JobScheduler( const QList<JobData>& lstJobs, QObject *_parent )
    : QObject( _parent )
    , m_jobSchedule()
{
    for( const auto& jobItem : lstJobs )
    {
        this->createJob( jobItem );
    }
}

void JobScheduler::slotRunJob( const QString& jobName )
{
    LogInfo( "JobScheduler::slotRunJob", QString( "SlotRunJob fired for job : %1" ).arg( jobName ) );
    if( m_jobSchedule.contains( jobName ) )
    {
        emit signalRunJob( m_jobSchedule.value( jobName )->jobData() );
    }

    this->rescheduleJob( jobName );
}

void JobScheduler::scheduleJob( const QString& _job_name,
                                const QHash<QString, QVariant>& _param_list,
                                const int _interval,
                                const QString& _target_object,
                                const QTime& _run_time,
                                const QDate& _run_date )
{

    // Get the corresponding job from the map.
    QPointer<JobTimer> schedJob = m_jobSchedule.value( _job_name );
    if( nullptr == schedJob )
    {
        LogInfo( "[JobScheduler::scheduleJob]", QString( "Job %1 not added to the an unknown error." ).arg( _job_name ) );
        return;
    }

    // Scheduling is done within 24 Hours. After each timer-fire we check if the rundate is equal to the currentDate.
    // If so.. Run the job. If not, reschedule for the next 24 hours.
    if( _run_time.isValid() || _run_date.isValid() )
    {
        QDateTime schedDT( QDate::currentDate(), _run_time );

        if( QDateTime::currentDateTime().secsTo( schedDT ) <= 0 )
        {
            // Add one day from today.
            schedDT = schedDT.addDays( 1 );
        }

        // Check if the jobName already exists. If so, re-schedule the existing job. If not, create the job.
        if( !m_jobSchedule.contains( _job_name ) )
        {
            this->createJob( JobData( _job_name, _param_list, _run_time, _interval, _target_object, _run_date ) );
        }

        if( 0 == _interval )
        {
            qint64 secondsToRun = QDateTime::currentDateTime().secsTo( schedDT );
            schedJob->start( static_cast<int>( secondsToRun * 1000 ) );
        }
    }
    else if( 0 < _interval )
    {
        schedJob->start( _interval * 1000 );
    }

    if( schedJob->isActive() )
    {
        LogInfo( "[JobScheduler::scheduleJob]", QString( "Job : %1 scheduled to run in %2 seconds ( Date : %3, Time : %4 )" )
                                                .arg( _job_name )
                                                .arg( schedJob->remainingTime() / 1000 )
                                                .arg( QDateTime::currentDateTime().addSecs( schedJob->remainingTime() / 1000 ).date().toString() )
                                                .arg( QDateTime::currentDateTime().addSecs( schedJob->remainingTime() / 1000 ).time().toString() ) );
    }
    else
    {
        LogError( "[JobScheduler::scheduleJob]", QString( "Job : %1 failed to schedule." )
                                                .arg( _job_name ) );

    }
}

void JobScheduler::rescheduleJob ( const QString& _job_name )
{
    if( m_jobSchedule.contains( _job_name ) )
    {
        QPointer<JobTimer> schedJob = m_jobSchedule.value( _job_name );
        if( nullptr == schedJob )
        {
            LogInfo( "[JobScheduler::rescheduleJob]", QString( "Job %1 not rescheduled to the an unknown error." ).arg( _job_name ) );
            return;
        }

        this->scheduleJob( _job_name,
                           schedJob->jobData().paramList(),
                           schedJob->jobData().runInterval(),
                           schedJob->jobData().targetObject(),
                           schedJob->jobData().runTime(),
                           schedJob->jobData().runDate() );
    }
    else
    {
        LogError( "[JobScheduler::scheduleJob]", QString( "Unable to re-schedule job %1 as it is unknown." ).arg( _job_name ) );
    }
}

void JobScheduler::createJob( const JobData& _jobItem )
{
    QPointer<JobTimer> newJob = new JobTimer( _jobItem, this );    // Create a new job
    m_jobSchedule.insert( _jobItem.jobName(), newJob );      // Add to the list
    QObject::connect( newJob.data(), &JobTimer::signalRunJob, this, &JobScheduler::slotRunJob );
}

void JobScheduler::start()
{
    // Schedule all stored jobs.
    for( const QString& _jobName : QStringList( m_jobSchedule.keys() ) )
    {
        this->rescheduleJob( _jobName );
    }
}
