/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
// Local
#include "pingdata.h"
#include "compatqt514.h"

// Qt
#include <QtDebug>

using namespace osdev::components;

PingData::PingData()
    : m_time_stamp()
    , m_host_name()
    , m_ip_address()
    , m_packet_loss( 0 )
    , m_rtt_min( 0 )
    , m_rtt_avg( 0 )
    , m_rtt_max( 0 )
    , m_rtt_mdev( 0 )
    , m_available( false )
{

}

PingData::PingData( const QString &host_name, const QStringList &output_data )
    : m_time_stamp()
    , m_host_name( host_name )
    , m_ip_address()
    , m_packet_loss( 0 )
    , m_rtt_min( 0 )
    , m_rtt_avg( 0 )
    , m_rtt_max( 0 )
    , m_rtt_mdev( 0 )
    , m_available( false )
{
    parseOutputData( output_data );
}

PingData::~PingData()
{

}

void PingData::parseOutputData( const QStringList &output_data )
{
    m_time_stamp = QDateTime::currentDateTime();
    if( output_data.size() == 5 )
    {
        // -------------------------------------------------------------------------
        // Get the IP-Address from the structure
        // Format of the line : PING osdev.nl (x.x.x.x) 56(84) bytes of data.
        QString ipAddr = output_data.at(1).split( " ", SKIP_EMPTY_PARTS ).at( 2 );
        m_ip_address = ipAddr.replace( "(", "" ).replace( ")", "" );

        // -------------------------------------------------------------------------
        // Get the packet losses
        if( output_data.at( 3 ).split( " ", SKIP_EMPTY_PARTS ).size() < 12 )
        {
            QString s_losses = output_data.at( 3 ).split( " ", SKIP_EMPTY_PARTS ).at( 5 );
            m_packet_loss = s_losses.replace( "%", "" ).toInt();
        }
        else if( output_data.at( 3 ).split( " ", SKIP_EMPTY_PARTS ).size() > 2 )
        {
            QString s_losses = output_data.at( 3 ).split( " ", SKIP_EMPTY_PARTS ).at( 7 );
            m_packet_loss = s_losses.replace( "%", "" ).toInt();
        }

        // -------------------------------------------------------------------------
        // Get the min/max statistics.
        if( output_data.at( 4 ).split( " ", SKIP_EMPTY_PARTS ).size() > 2 )
        {
            QString s_statline = output_data.at( 4 ).split( " ", SKIP_EMPTY_PARTS ).at( 3 );
            QStringList s_lstData = s_statline.split( "/", SKIP_EMPTY_PARTS );

            m_rtt_min  = s_lstData.at( 0 ).toFloat();
            m_rtt_avg  = s_lstData.at( 1 ).toFloat();
            m_rtt_max  = s_lstData.at( 2 ).toFloat();
            m_rtt_mdev = s_lstData.at( 3 ).toFloat();
        }

        if( output_data.at( 3 ).contains( "100% packet loss" ) )
        {
            m_available = false;
        }
        else
        {
            m_available = true;
        }
    }
    else if( output_data.size() == 4 )
    {
        // Get the IP-Address
        QString ipAddr = output_data.at( 1 ).split( " ", SKIP_EMPTY_PARTS ).at( 2 );
        m_ip_address = ipAddr.replace( "(", "" ).replace( ")", "" );

        // Get the packet losses
        QString s_losses = output_data.at( 3 ).split( " ", SKIP_EMPTY_PARTS ).at( 5 );
        m_packet_loss = s_losses.replace( "%", "" ).toInt();

        m_available = false;
    }
}


QString PingData::asString()
{
    QString sResult;

    sResult += QString( "Hostname   : %1\n" ).arg( m_host_name );
    sResult += QString( "\t|-> IpAddress    : %1\n" ).arg( m_ip_address );
    sResult += QString( "\t|-> Packet loss  : %1\n" ).arg( m_packet_loss );
    sResult += QString( "\t|-> RTT min      : %1\n" ).arg( m_rtt_min );
    sResult += QString( "\t|-> RTT max      : %1\n" ).arg( m_rtt_max );
    sResult += QString( "\t|-> RTT average  : %1\n" ).arg( m_rtt_avg );
    sResult += QString( "\t|-> RTT mdev     : %1\n" ).arg( m_rtt_mdev );
    sResult += QString( "\t|-> Available    : " );
    if( m_available )
    {
        sResult += QString( "yes\n" );
    }
    else
    {
        sResult += QString( "no\n" );
    }

    return sResult;
}
