/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include "httpclient.h"
#include "log.h"

using namespace osdev::components;

HttpClient::HttpClient( QObject *parent )
    : QObject( parent )
    , m_pNetworkManager( new QNetworkAccessManager( this ) )
{
    this->connectSignalsToSlots();
}

void HttpClient::connectSignalsToSlots()
{
    connect( m_pNetworkManager.data(), &QNetworkAccessManager::finished, this, &HttpClient::slotHttpRequestFinished );
}

void HttpClient::sendRequest( const QString &ip_address, const QString &argument )
{
    QUrl l_url = QString( "http://" + ip_address + argument );
    m_pNetworkManager->get( QNetworkRequest( l_url ) );
}

void HttpClient::slotHttpRequestFinished( QNetworkReply *data )
{
    if( data )
    {
        switch( data->error() )
        {
            case QNetworkReply::NoError:
            {
                QString l_data = data->readAll();
                emit signalDataReceived( data->url().toString(), l_data, this );
                break;
            }
            case QNetworkReply::TimeoutError:
            {
                // Set all data to '0' to indicate the sensor offline
                break;
            }
            case QNetworkReply::ConnectionRefusedError:
            case QNetworkReply::RemoteHostClosedError:
            case QNetworkReply::HostNotFoundError:
            case QNetworkReply::OperationCanceledError:
            case QNetworkReply::SslHandshakeFailedError:
            case QNetworkReply::TemporaryNetworkFailureError:
            case QNetworkReply::NetworkSessionFailedError:
            case QNetworkReply::BackgroundRequestNotAllowedError:
#if(QT_VERSION > QT_VERSION_CHECK( 5, 6, 0) )
            case QNetworkReply::TooManyRedirectsError:
            case QNetworkReply::InsecureRedirectError:
#endif
            case QNetworkReply::ProxyConnectionRefusedError:
            case QNetworkReply::ProxyConnectionClosedError:
            case QNetworkReply::ProxyNotFoundError:
            case QNetworkReply::ProxyTimeoutError:
            case QNetworkReply::ProxyAuthenticationRequiredError:
            case QNetworkReply::ContentAccessDenied:
            case QNetworkReply::ContentOperationNotPermittedError:
            case QNetworkReply::ContentNotFoundError:
            case QNetworkReply::AuthenticationRequiredError:
            case QNetworkReply::ContentReSendError:
            case QNetworkReply::ContentConflictError:
            case QNetworkReply::ContentGoneError:
            case QNetworkReply::InternalServerError:
            case QNetworkReply::OperationNotImplementedError:
            case QNetworkReply::ServiceUnavailableError:
            case QNetworkReply::ProtocolUnknownError:
            case QNetworkReply::ProtocolInvalidOperationError:
            case QNetworkReply::UnknownNetworkError:
            case QNetworkReply::UnknownProxyError:
            case QNetworkReply::UnknownContentError:
            case QNetworkReply::ProtocolFailure:
            case QNetworkReply::UnknownServerError:
            {
                LogWarning( "[HttpClient::slotHttpRequestFinished]",
                           QString( "The following error occured : %1. Host : %2" )
                                    .arg( data->errorString() )
                                    .arg( data->url().host() ) );
                break;
            }
        }
    }
}
