/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_AUTODISCOVER_H
#define OSDEV_COMPONENTS_AUTODISCOVER_H

// Qt
#include <QObject>
#include <QString>
#include <QStringList>
#include <QPointer>

// Local
#include "pingmanager.h"
/*
 _______________________________________
/ Dealing with failure is easy:         \
|                                       |
| Work hard to improve. Success is also |
| easy to handle:                       |
|                                       |
| You've solved the wrong problem.      |
|                                       |
\ Work hard to improve.                 /
 ---------------------------------------
       .--.
      |o_o |
      |:_/ |
     //   \ \
    (|     | )
   /'\_   _/`\
   \___)=(___/
*/

namespace osdev {
namespace components {

/*!
 * \brief   Autodiscover will check the NIC for its networkaddress and scans
 *          the network for connected and active sensors.
 */
class AutoDiscover : public QObject
{
    Q_OBJECT

public:
    /*!
     * \brief AutoDiscover  - The default Constructor
     * \param parent        - The calling QObject
     */
    explicit AutoDiscover( QObject *parent = nullptr );

public slots:
    /*!
     * \brief startDiscovery - Start scanning the network for active sensors.
     */
    void startDiscovery();

private slots:
    /*!
     * \brief slotPingThreadsReady - Will be called when all threads are ready
     */
    void slotPingThreadsReady();

    /*!
     * \brief slotDataReceived - Called by the thread that received data.
     * \param ip_address - The ip-address of the sensor
     * \param data       - The data received
     * \param client     - The pointer to the calling object. Can be NULL in most cases.
     */
    void slotDataReceived( const QString &ip_address, const QString &data, QObject *client = nullptr );

signals:
    /*!
     * \brief signalSensorFound - Signal will fire when data was received from one of the threads.
     * \param data              - The received data.
     */
    void signalSensorFound( const QString &data );

private:
    /// Determine the network subnet(s) we're living in.
    void getNetworks();
    /// CHeck all found hosts if they respond to the ID query
    void checkHost();

private:    // Members ( Giggity! )
    QStringList     m_hosts_found;
    QStringList     m_hosts_to_check;

    PingManager     m_pingManager;

};

}       /* End namespace components */
}       /* End namespace osdev  */

#endif  /* OSDEV_COMPONENTS_AUTODISCOVER_H */
