/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
// Local
#include "autodiscover.h"
#include "log.h"
#include "httpclient.h"

// Qt
#include <QCoreApplication>
#include <QNetworkInterface>
#include <QHostAddress>

#include <QtDebug>

using namespace osdev::components;

AutoDiscover::AutoDiscover( QObject *parent )
    : QObject( parent )
    , m_hosts_found( QStringList() )
    , m_hosts_to_check( QStringList() )
    , m_pingManager( 10 )
{
    // Connect the PingManager to this object
    connect( &m_pingManager, &PingManager::signalAllThreadsFinished, this, &AutoDiscover::slotPingThreadsReady );

    // Start the network discovery
    this->startDiscovery();
}

void AutoDiscover::getNetworks()
{
    // Make sure we start clean and fresh.
    m_hosts_to_check.clear();
    LogInfo( "[AutoDiscover::getNetworks]", QString( "Starting with %1 hosts in the buffer" ).arg( m_hosts_to_check.count() ) );
    foreach( const QHostAddress &address, QNetworkInterface::allAddresses() )
    {
        if( address.protocol() == QAbstractSocket::IPv4Protocol
            && address != QHostAddress( QHostAddress::LocalHost ) )
        {
            LogInfo( "[AutoDiscover::getNetworks]", QString( "Found network address : %1" ).arg( address.toString() ) );
            QStringList octets = address.toString().split( "." );
            if( octets.size() == 4 )
            {
                for( int host = 1; host < 255; host++ )
                {
                    m_hosts_to_check.append( QString( "%1.%2.%3.%4" )
                                                .arg( octets.at( 0 ) )
                                                .arg( octets.at( 1 ) )
                                                .arg( octets.at( 2 ) )
                                                .arg( host ) );
                }
            }
        }
    }
}

void AutoDiscover::slotPingThreadsReady()
{
    LogInfo( "[AutoDiscover::slotPingThreadsReady]", QString( "Wrapping up the pings. Getting all Active IP-addresses" ) );
    // All Threads are done, which means the pingDataQueue is ready for reading.
    QStringList ip_lijst = m_pingManager.getActiveIPs();

    for( const auto &ip : ip_lijst )
    {
        LogInfo( "[AutoDiscover::slotPingThreadsReady]",
                QString( "Host : %1 is active." ) .arg( ip ) );

        HttpClient *pClient = new HttpClient( this );
        connect( pClient, &HttpClient::signalDataReceived, this, &AutoDiscover::slotDataReceived );
        pClient->sendRequest( ip, "/id" );
    }

}

void AutoDiscover::startDiscovery()
{
    // Produce all nodes within this network
    getNetworks();
    // Check if the found hosts are actually sensors.
    checkHost();
}

void AutoDiscover::checkHost()
{
    LogInfo( "[AutoDiscover::checkHost]", QString( "Adding %1 hosts to the pingmanager" ).arg( m_hosts_to_check.count() ) );
    m_pingManager.addHosts( m_hosts_to_check );
    m_pingManager.startPingTests();
}

void AutoDiscover::slotDataReceived( const QString &ip_address, const QString &data, QObject *client )
{
    if( data.contains( "CO2MOS" ) )
    {
        LogInfo( "[AutoDiscover::slotDataReceived]", QString( "Host : %1 seems to be a CO2MOS Smart Sensor. Data Received : %2" )
                                                        .arg( ip_address )
                                                        .arg( data ) );

        // Send this ip-address with String, Id and IP-address, separated by colons.
        emit signalSensorFound( data );
    }
    else
    {
        LogInfo( "[AutoDiscover::slotDataReceived]", QString( "Host : %1 doesn't seem to be a CO2 Sensor." ).arg( ip_address ) );
    }

    if( client )
    {
        LogInfo( "[AutoDiscover::slotDataReceived]", QString( "Deleting the ClientObject will be done if the parent object is removed." ) );
        // delete client;
    }
}
