/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_SYSTEMINFO_H
#define OSDEV_COMPONENTS_SYSTEMINFO_H

#include "globallibexport.h"

#include <memory>
#include <QString>

namespace osdev  {
namespace components {

/**
 * @brief Collects information about the current system
 *
 * Singleton class.
 */
class GLOBALLIBINTERFACE SystemInfo
{
public:
    /// Supported classes of configuration file
    enum class ConfigFile {
        System,     ///< System configuration
        UI,         ///< UI configuration
        Backend     ///< Backend configuration
    };

    //! Constructs the systeminfo (Ctor).
    //! @return Reference to the SystemInfo instance
    static SystemInfo& Instance();

    /// Deleted copy-constructor
    SystemInfo(const SystemInfo&) = delete;
    /// Deleted assignment operator
    SystemInfo& operator= (const SystemInfo&) = delete;
    /// Deleted move-constructor
    SystemInfo(SystemInfo&&) = delete;
    /// Deleted move operator
    SystemInfo& operator=(SystemInfo&&) = delete;

    //! Destructs the config parser (Dtor)
    ~SystemInfo();

    //! Destroy single instance
    static void destroyInstance();

    //! Set the application path.
    void setApplicationPath(const QString& qsName);

    //! Returns the application path.
    QString getApplicationPath() const;

    //! Returns the location (path to) the executable
    QString getExecutablePath() const;

    //! Return xml configuration path.
    QString getConfigurationPath() const;

    //! Return resource path.
    QString getResourcePath() const;

    //! Returns the path of a config file.
    QString getConfigFile( ConfigFile fileType ) const;

private:
    //! Constructs the config parser (Ctor)
    SystemInfo();

private:

    //! Contains the only instance of a CSystemInfo.
    static std::unique_ptr<SystemInfo> s_pInstance;

    //! The name of the configfile.
    QString m_qsApplicationPath;

    //! The path with the location of the executable
    QString m_qsExecutablePath;
};

} // End namespace components
} // End namespace osdev

#endif /* OSDEV_COMPONENTS_SYSTEMINFO_H */
